(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[    225700,       7680]*)
(*NotebookOutlinePosition[    230819,       7808]*)
(*  CellTagsIndexPosition[    230671,       7800]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["\<\
Chapter 3
Systems with Quadratic and Cubic Nonlinearities\
\>", "Title",
  TextAlignment->Left,
  TextJustification->0],

Cell[BoxData[
    \(Off[General::spell1]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell["3.1 Nondimensional Equation of Motion", "Section",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We consider the free oscillations of a particle of mass ",
  StyleBox["m",
    FontSlant->"Italic"],
  " under the action of gravity and restrained by a nonlinear spring. The \
equation of motion is"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq31a = 
      m\ D[\(x\^*\)[\(t\^*\)], {\(t\^*\), 2}] + f[\(x\^*\)[\(t\^*\)]] == 
        m\ g\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(f[\(x\^*\)[\(t\^*\)]]\), "+", 
        RowBox[{"m", " ", 
          RowBox[{
            SuperscriptBox[\((\(x\^*\))\), "\[DoublePrime]",
              MultilineFunction->None], "[", \(t\^*\), "]"}]}]}], "==", 
      \(g\ m\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`g\)]],
  " is the gravitational acceleration and ",
  Cell[BoxData[
      \(TraditionalForm\`f[\(x\^*\)]\)]],
  " is the restraining force of the spring. We assume that ",
  Cell[BoxData[
      \(TraditionalForm\`f[\(x\^*\)]\)]],
  " is an odd cubic function of ",
  Cell[BoxData[
      \(TraditionalForm\`\(\(x\^*\); \)\)]],
  " that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq31b = eq31a /. f[a_] -> k\_1\ a + k\_3\ a\^3\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(k\_1\ \(x\^*\)[\(t\^*\)]\), "+", \(k\_3\ \(x\^*\)[\(t\^*\)]\^3\), 
        "+", 
        RowBox[{"m", " ", 
          RowBox[{
            SuperscriptBox[\((\(x\^*\))\), "\[DoublePrime]",
              MultilineFunction->None], "[", \(t\^*\), "]"}]}]}], "==", 
      \(g\ m\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The equilibrium positions ",
  Cell[BoxData[
      \(TraditionalForm\`\(x\^*\) = \(\(x\_s\%*\) = constant\)\)]],
  " can be obtained by dropping the acceleration term. The result is"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq31c = eq31b /. \(x\^*\) -> \((\(x\_s\%*\)\ &)\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(k\_1\ \(\((x\_s)\)\^*\) + k\_3\ \((\(\((x\_s)\)\^*\))\)\^3 == g\ m\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
In this chapter, we investigate small oscillations about one of the \
equilibrium positions. To this end, we shift this equilibrium position to the \
origin by using the transformation\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(xRule = {\(x\^*\) -> \((\(x\_s\%*\) + \(u\^*\)[#]\ &)\)}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting this transformation into ",
  StyleBox["eq31b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and using the equilibrium condition ",
  StyleBox["eq31c",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq31d = 
      Expand[Subtract\ @@\ eq31b /. xRule] == 0 /. 
        \((eq31c /. Equal -> Rule)\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(k\_1\ \(u\^*\)[\(t\^*\)]\), "+", 
        \(3\ k\_3\ \((\(\((x\_s)\)\^*\))\)\^2\ \(u\^*\)[\(t\^*\)]\), "+", 
        \(3\ k\_3\ \(\((x\_s)\)\^*\)\ \(u\^*\)[\(t\^*\)]\^2\), "+", 
        \(k\_3\ \(u\^*\)[\(t\^*\)]\^3\), "+", 
        RowBox[{"m", " ", 
          RowBox[{
            SuperscriptBox[\((\(u\^*\))\), "\[DoublePrime]",
              MultilineFunction->None], "[", \(t\^*\), "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
As before, we introduce the following dimensionless quantities:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(dimenRule = {\(u\^*\) -> \((\(x\_s\%*\)\ u[#/T]\ &)\), 
        \(t\^*\) -> T\ t}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "into ",
  StyleBox["eq31d",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq31e = 
      \((eq31d[\([1]\)]\ T\^2/\((m\ \(x\_s\%*\))\) /. dimenRule // Expand)
          \) == 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(\(T\^2\ k\_1\ u[t]\)\/m\), "+", 
        \(\(3\ T\^2\ k\_3\ \((\(\((x\_s)\)\^*\))\)\^2\ u[t]\)\/m\), "+", 
        \(\(3\ T\^2\ k\_3\ \((\(\((x\_s)\)\^*\))\)\^2\ u[t]\^2\)\/m\), "+", 
        \(\(T\^2\ k\_3\ \((\(\((x\_s)\)\^*\))\)\^2\ u[t]\^3\)\/m\), "+", 
        RowBox[{
          SuperscriptBox["u", "\[DoublePrime]",
            MultilineFunction->None], "[", "t", "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "We rewrite ",
  StyleBox["eq31e",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq31f = 
      \((Collect[eq31e[\([1]\)], 
              u[t]] /. {_\ u[t]^n_.  -> \[Alpha]\_n\ u[t]^n})\) == 0\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(\[Alpha]\_1\ u[t]\), "+", \(\[Alpha]\_2\ u[t]\^2\), "+", 
        \(\[Alpha]\_3\ u[t]\^3\), "+", 
        RowBox[{
          SuperscriptBox["u", "\[DoublePrime]",
            MultilineFunction->None], "[", "t", "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where the ",
  Cell[BoxData[
      \(TraditionalForm\`\[Alpha]\_j\)]],
  " are given by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(alphas = 
      \(\((\[Alpha]\_# -> Coefficient[eq31e[\([1]\)], u[t]\^#])\)\ &\)\ /@
        \ {1, 2, 3}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Alpha]\_1 \[Rule] 
        \(T\^2\ k\_1\)\/m + \(3\ T\^2\ k\_3\ \((\(\((x\_s)\)\^*\))\)\^2\)\/m, 
      \[Alpha]\_2 \[Rule] \(3\ T\^2\ k\_3\ \((\(\((x\_s)\)\^*\))\)\^2\)\/m, 
      \[Alpha]\_3 \[Rule] \(T\^2\ k\_3\ \((\(\((x\_s)\)\^*\))\)\^2\)\/m}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "We choose ",
  Cell[BoxData[
      \(TraditionalForm\`T\)]],
  " so that ",
  Cell[BoxData[
      \(\[Alpha]\_1 = 1\)]],
  " and hence the natural frequency is unity; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(TRule = \(Solve[\[Alpha]\_1 == 1 /. alphas, T]\)[\([2]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({T \[Rule] \@m\/\@\(k\_1 + 3\ k\_3\ \((\(\((x\_s)\)\^*\))\)\^2\)}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Consequently, ",
  Cell[BoxData[
      \(TraditionalForm\`\[Alpha]\_2\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Alpha]\_3\)]],
  " can be rewritten as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(alphas = alphas /. TRule // Rest\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Alpha]\_2 \[Rule] 
        \(3\ k\_3\ \((\(\((x\_s)\)\^*\))\)\^2\)\/\(k\_1 + 
            3\ k\_3\ \((\(\((x\_s)\)\^*\))\)\^2\), 
      \[Alpha]\_3 \[Rule] 
        \(k\_3\ \((\(\((x\_s)\)\^*\))\)\^2\)\/\(k\_1 + 
            3\ k\_3\ \((\(\((x\_s)\)\^*\))\)\^2\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "With ",
  Cell[BoxData[
      \(TraditionalForm\`\[Alpha]\_1\)]],
  " being unity, ",
  StyleBox["eq31f ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "becomes"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq31g = eq31f /. \[Alpha]\_1 -> 1\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(u[t]\), "+", \(\[Alpha]\_2\ u[t]\^2\), "+", \(\[Alpha]\_3\ u[t]\^3\), 
        "+", 
        RowBox[{
          SuperscriptBox["u", "\[DoublePrime]",
            MultilineFunction->None], "[", "t", "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "In contrast with the ",
  StyleBox["DuffingEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " contains a quadratic as well as a cubic term."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In the next section, we determine a second-order straightforward expansion \
to the solutions of ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " for small but finite amplitudes. In Section 3.3, we determine a uniform \
second-order expansion by using the Lindstedt-Poincar\[EAcute] technique. In \
Section 3.4, we determine a uniform second-order expansion by using the \
method of multiple scales. In Section 3.5, we show that the first \
approximation obtained with the method of averaging yields an incomplete \
solution. In Section 3.6, we introduce the generalized method of averaging \
and obtain a uniform second-order expansion. Finally, in Sections 3.7 and \
3.8, we introduce the Krylov-Bogoliubov-Mitropolsky technique and the method \
of normal forms, respectively."
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["3.2 Straightforward Expansion", "Section",
  CellMargins->{{Inherited, 17.4375}, {Inherited, Inherited}},
  Evaluatable->False,
  TextAlignment->Left,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],

Cell[TextData[{
  "To carry out a straightforward expansion for small but finite amplitudes \
for ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we need to introduce a small parameter \[Epsilon] as a bookkeeping \
parameter. In terms of this parameter, we seek a third-order expansion in the \
form"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(uExpRule[maxOrder_] := 
      u -> \((Sum[\[Epsilon]\^i\ u\_i[#], {i, 1, maxOrder}]\ &)\)\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where maxOrder is the order of the expansion sought. In what follows, we \
let maxOrder",
  Cell[BoxData[
      \(TraditionalForm\`\( = 3\)\)]],
  " for an expansion of order three. Clearly, \[Epsilon] is a measure of the \
amplitude of oscillation and can be set equal to unity in the final solution \
if the amplitude is taken to be small, as described below. Substituting this \
expansion into ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and discarding terms of order \
higher than ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^3\)]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq32a = 
      \((eq31g /. uExpRule[3] // ExpandAll)\) /. 
        \[Epsilon]\^\(n_ /; n > 3\) -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(\[Epsilon]\ u\_1[t]\), "+", 
        \(\[Epsilon]\^2\ \[Alpha]\_2\ u\_1[t]\^2\), "+", 
        \(\[Epsilon]\^3\ \[Alpha]\_3\ u\_1[t]\^3\), "+", 
        \(\[Epsilon]\^2\ u\_2[t]\), "+", 
        \(2\ \[Epsilon]\^3\ \[Alpha]\_2\ u\_1[t]\ u\_2[t]\), "+", 
        \(\[Epsilon]\^3\ u\_3[t]\), "+", 
        RowBox[{"\[Epsilon]", " ", 
          RowBox[{
            SubsuperscriptBox["u", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "t", "]"}]}], "+", 
        RowBox[{\(\[Epsilon]\^2\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "t", "]"}]}], "+", 
        RowBox[{\(\[Epsilon]\^3\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "3", "\[DoublePrime]",
              MultilineFunction->None], "[", "t", "]"}]}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Equating coefficients of like powers of \[Epsilon] in ",
  StyleBox["eq32a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqEps = 
      Table[Coefficient[eq32a[\([1]\)], \[Epsilon]\^i] == 0, {i, 3}]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(u\_1[t]\), "+", 
            RowBox[{
              SubsuperscriptBox["u", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "t", "]"}]}], "==", "0"}], 
        ",", 
        RowBox[{
          RowBox[{\(\[Alpha]\_2\ u\_1[t]\^2\), "+", \(u\_2[t]\), "+", 
            RowBox[{
              SubsuperscriptBox["u", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "t", "]"}]}], "==", "0"}], 
        ",", 
        RowBox[{
          RowBox[{
          \(\[Alpha]\_3\ u\_1[t]\^3\), "+", 
            \(2\ \[Alpha]\_2\ u\_1[t]\ u\_2[t]\), "+", \(u\_3[t]\), "+", 
            RowBox[{
              SubsuperscriptBox["u", "3", "\[DoublePrime]",
                MultilineFunction->None], "[", "t", "]"}]}], "==", "0"}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "which can be solved sequentially for ",
  Cell[BoxData[
      \(TraditionalForm\`u\_1, u\_2, \)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`u\_3\)]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "The general solution of the first-order equation, ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", can be expressed as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol1 = 
      \(DSolve[{eqEps[\([1]\)], u\_1[0] == a\ Cos[\[Beta]], 
              \(u\_1\^\[Prime]\)[0] == \(-a\)\ Sin[\[Beta]]}, u\_1[t], t]\)[
          \([1]\)] // Simplify\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_1[t] \[Rule] a\ Cos[t + \[Beta]]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`a\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Beta]\)]],
  " are constants. Substituting ",
  StyleBox["sol1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`u\_1[t]\)]],
  ", into the second-order equation, ",
  StyleBox["eqEps[[2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and solving the resulting equation for ",
  Cell[BoxData[
      \(TraditionalForm\`u\_2[t]\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol2 = 
      \(DSolve[eqEps[\([2]\)] /. sol1, u\_2[t], t]\)[\([1]\)] /. C[_] -> 0 // 
        Simplify\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_2[t] \[Rule] 
        1\/6\ a\^2\ \((\(-3\) + Cos[2\ \((t + \[Beta])\)])\)\ \[Alpha]\_2}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "As before, we do not include the solution of the homogeneous problem for ",
  
  Cell[BoxData[
      \(TraditionalForm\`u\_2[t]\)]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["sol1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["sol2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the third-order equation, ",
  StyleBox["eqEps[[3]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and solving the resulting equation ",
  Cell[BoxData[
      \(TraditionalForm\`u\_3[t]\)]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol3 = 
      \(DSolve[\(eqEps[\([3]\)] /. sol1\) /. sol2, u\_3[t], t]\)[\([1]\)] /. 
          C[_] -> 0 // Simplify\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_3[t] \[Rule] 
        1\/96\ a\^3\ 
          \((2\ \((
                  10\ Cos[t + \[Beta]] + Cos[3\ \((t + \[Beta])\)] + 
                    20\ t\ Sin[t + \[Beta]])\)\ \[Alpha]\_2\%2 + 
              3\ \((\(-6\)\ Cos[t + \[Beta]] + Cos[3\ \((t + \[Beta])\)] - 
                    12\ t\ Sin[t + \[Beta]])\)\ \[Alpha]\_3)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Simply letting ",
  Cell[BoxData[
      \(TraditionalForm\`C[_] \[RightArrow] 0\)]],
  " did not remove all of the solutions of the homogeneous equation in this \
case. Therefore, we clear it one more time and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol3 = sol3 /. Cos[t + _] -> 0 // ExpandAll\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_3[t] \[Rule] 
        1\/48\ a\^3\ Cos[3\ t + 3\ \[Beta]]\ \[Alpha]\_2\%2 + 
          5\/12\ a\^3\ t\ Sin[t + \[Beta]]\ \[Alpha]\_2\%2 + 
          1\/32\ a\^3\ Cos[3\ t + 3\ \[Beta]]\ \[Alpha]\_3 - 
          3\/8\ a\^3\ t\ Sin[t + \[Beta]]\ \[Alpha]\_3}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Combining the first-, second-, and third-order solutions, we obtain, to the \
third approximation, that\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol = 
      u[t] == \((\(u[t] /. uExpRule[3]\) /. Flatten[{sol1, sol2, sol3}])
          \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u[t] == 
      a\ \[Epsilon]\ Cos[t + \[Beta]] + 
        1\/6\ a\^2\ \[Epsilon]\^2\ \((\(-3\) + Cos[2\ \((t + \[Beta])\)])\)\ 
          \[Alpha]\_2 + 
        \[Epsilon]\^3\ 
          \((1\/48\ a\^3\ Cos[3\ t + 3\ \[Beta]]\ \[Alpha]\_2\%2 + 
              5\/12\ a\^3\ t\ Sin[t + \[Beta]]\ \[Alpha]\_2\%2 + 
              1\/32\ a\^3\ Cos[3\ t + 3\ \[Beta]]\ \[Alpha]\_3 - 
              3\/8\ a\^3\ t\ Sin[t + \[Beta]]\ \[Alpha]\_3)\)\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "We note that the dependence of ",
  StyleBox["u",
    FontSlant->"Italic"],
  " on \[Epsilon] and ",
  StyleBox["a",
    FontSlant->"Italic"],
  " is in the combination ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\ a\)]],
  ". Thus, one can set ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon] = 1\)]],
  " and consider ",
  StyleBox["a",
    FontSlant->"Italic"],
  " as the perturbation parameter."
}], "Text",
  CellMargins->{{13, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "The straightforward expansion ",
  StyleBox["sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " breaks down for ",
  Cell[BoxData[
      \(TraditionalForm
      \`t \[GreaterEqual] O(\(\[Epsilon]\^\(-1\)\) a\^\(-1\))\)]],
  " because the third-order term is the same order or larger than the \
second-order term owing to the presence of the mixed-secular term. Next, we \
use the Lindstedt-Poincar\[EAcute] technique to determine a third-order \
uniform solution."
}], "Text",
  CellMargins->{{13, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["3.3 The Lindstedt-Poincar\[EAcute] Technique", "Section",
  CellMargins->{{Inherited, 17.4375}, {Inherited, Inherited}},
  Evaluatable->False,
  TextAlignment->Left,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],

Cell[TextData[{
  "To apply the Lindstedt-Poincar\[EAcute] technique to ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we introduce the transformation ",
  Cell[BoxData[
      \(TraditionalForm\`\[Tau] = \[Omega]\ t\)]],
  " and rewrite it as"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq33a = 
      \(eq31g /. u -> \((u[\[Omega]\ #]\ &)\)\) /. t -> \[Tau]/\[Omega]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(u[\[Tau]]\), "+", \(\[Alpha]\_2\ u[\[Tau]]\^2\), "+", 
        \(\[Alpha]\_3\ u[\[Tau]]\^3\), "+", 
        RowBox[{\(\[Omega]\^2\), " ", 
          RowBox[{
            SuperscriptBox["u", "\[DoublePrime]",
              MultilineFunction->None], "[", "\[Tau]", "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine a uniform expansion of order maxOrder of the solution of ",
  StyleBox["eq33a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we expand both \[Omega] and ",
  Cell[BoxData[
      \(TraditionalForm\`u[\[Tau]]\)]],
  " in powers of \[Epsilon] as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[{
    \(omgRule[maxOrder_] := 
      \[Omega] -> 1 + Sum[\[Epsilon]\^i\ \[Omega]\_i, {i, maxOrder}]\), 
    \(uExpRule[maxOrder_] := 
      u -> \((Sum[\[Epsilon]\^i\ u\_i[#], {i, maxOrder + 1}]\ &)\)\)}], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "As discussed in the preceding chapter, the first term in the expansion of \
\[Omega] is taken to be the linear natural frequency of the system, which is \
unity in this case. We note that the order of the expansion for ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  " is larger than that of \[Omega] and the uniform expansion required. As \
discussed below, we eliminate the terms that lead to secular terms from the \
equation at order maxOrder+1 but do not include in the solution of the \
resulting equation in the expansion."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell["For a uniform second order expansion, we let", "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(maxOrder = 2; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the expansions of ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  " and \[Omega] into ",
  StyleBox["eq33a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and discarding terms of order \
higher than ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^3\)]],
  ", we obtain"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq33b = 
      \((eq33a /. {uExpRule[maxOrder], omgRule[maxOrder]} // ExpandAll)\) /. 
        \[Epsilon]\^\(n_ /; n > maxOrder + 1\) -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(\[Epsilon]\ u\_1[\[Tau]]\), "+", 
        \(\[Epsilon]\^2\ \[Alpha]\_2\ u\_1[\[Tau]]\^2\), "+", 
        \(\[Epsilon]\^3\ \[Alpha]\_3\ u\_1[\[Tau]]\^3\), "+", 
        \(\[Epsilon]\^2\ u\_2[\[Tau]]\), "+", 
        \(2\ \[Epsilon]\^3\ \[Alpha]\_2\ u\_1[\[Tau]]\ u\_2[\[Tau]]\), "+", 
        \(\[Epsilon]\^3\ u\_3[\[Tau]]\), "+", 
        RowBox[{"\[Epsilon]", " ", 
          RowBox[{
            SubsuperscriptBox["u", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "\[Tau]", "]"}]}], "+", 
        RowBox[{"2", " ", \(\[Epsilon]\^2\), " ", \(\[Omega]\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "\[Tau]", "]"}]}], "+", 
        RowBox[{\(\[Epsilon]\^3\), " ", \(\[Omega]\_1\%2\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "\[Tau]", "]"}]}], "+", 
        RowBox[{"2", " ", \(\[Epsilon]\^3\), " ", \(\[Omega]\_2\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "1", "\[DoublePrime]",
              MultilineFunction->None], "[", "\[Tau]", "]"}]}], "+", 
        RowBox[{\(\[Epsilon]\^2\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "\[Tau]", "]"}]}], "+", 
        RowBox[{"2", " ", \(\[Epsilon]\^3\), " ", \(\[Omega]\_1\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "2", "\[DoublePrime]",
              MultilineFunction->None], "[", "\[Tau]", "]"}]}], "+", 
        RowBox[{\(\[Epsilon]\^3\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "3", "\[DoublePrime]",
              MultilineFunction->None], "[", "\[Tau]", "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqEps = 
      Table[Coefficient[eq33b[\([1]\)], \[Epsilon]\^i] == 0, {i, 3}]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(u\_1[\[Tau]]\), "+", 
            RowBox[{
              SubsuperscriptBox["u", "1", "\[DoublePrime]",
                MultilineFunction->None], "[", "\[Tau]", "]"}]}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
          \(\[Alpha]\_2\ u\_1[\[Tau]]\^2\), "+", \(u\_2[\[Tau]]\), "+", 
            RowBox[{"2", " ", \(\[Omega]\_1\), " ", 
              RowBox[{
                SubsuperscriptBox["u", "1", "\[DoublePrime]",
                  MultilineFunction->None], "[", "\[Tau]", "]"}]}], "+", 
            RowBox[{
              SubsuperscriptBox["u", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "\[Tau]", "]"}]}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
          \(\[Alpha]\_3\ u\_1[\[Tau]]\^3\), "+", 
            \(2\ \[Alpha]\_2\ u\_1[\[Tau]]\ u\_2[\[Tau]]\), "+", 
            \(u\_3[\[Tau]]\), "+", 
            RowBox[{\(\[Omega]\_1\%2\), " ", 
              RowBox[{
                SubsuperscriptBox["u", "1", "\[DoublePrime]",
                  MultilineFunction->None], "[", "\[Tau]", "]"}]}], "+", 
            RowBox[{"2", " ", \(\[Omega]\_2\), " ", 
              RowBox[{
                SubsuperscriptBox["u", "1", "\[DoublePrime]",
                  MultilineFunction->None], "[", "\[Tau]", "]"}]}], "+", 
            RowBox[{"2", " ", \(\[Omega]\_1\), " ", 
              RowBox[{
                SubsuperscriptBox["u", "2", "\[DoublePrime]",
                  MultilineFunction->None], "[", "\[Tau]", "]"}]}], "+", 
            RowBox[{
              SubsuperscriptBox["u", "3", "\[DoublePrime]",
                MultilineFunction->None], "[", "\[Tau]", "]"}]}], "==", 
          "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The general solution of the first-order equation, ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", can be expressed as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol1 = 
      \(DSolve[{eqEps[\([1]\)], u\_1[0] == a\ Cos[\[Beta]], 
            \(u\_1\^\[Prime]\)[0] == \(-a\)\ Sin[\[Beta]]}, u\_1, \[Tau]]\)[
        \([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_1 \[Rule] 
        \((a\ Cos[\[Beta]]\ Cos[#1] - a\ Sin[\[Beta]]\ Sin[#1]&)\)}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`a\)]],
  " and \[Beta] are constants. Instead of using ",
  Cell[BoxData[
      \(TraditionalForm\`u\_1[\[Tau]]\)]],
  ", we use ",
  Cell[BoxData[
      \(TraditionalForm\`u\_1\)]],
  " in the second argument to ",
  StyleBox["DSolve",
    FontWeight->"Bold"],
  " so that ",
  StyleBox["sol1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be used in the terms involved with derivatives."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["sol1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`u\_1[\[Tau]]\)]],
  ", into the second-order equation, ",
  StyleBox["eqEps[[2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", moving the nonhomogeneous terms to the right-hand side of the resulting \
equation, we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"eq33c", "=", 
      RowBox[{
        RowBox[{
          RowBox[{\(u\_2[\[Tau]]\), "+", 
            RowBox[{
              SubsuperscriptBox["u", "2", "\[DoublePrime]",
                MultilineFunction->None], "[", "\[Tau]", "]"}], "-", "#"}], 
          " ", "&"}], " ", "/@", " ", 
        \(Reverse[eqEps[\([2]\)] /. sol1 // Simplify]\)}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(u\_2[\[Tau]]\), "+", 
        RowBox[{
          SubsuperscriptBox["u", "2", "\[DoublePrime]",
            MultilineFunction->None], "[", "\[Tau]", "]"}]}], "==", 
      \(\(-a\^2\)\ Cos[\[Beta] + \[Tau]]\^2\ \[Alpha]\_2 + 
        2\ a\ Cos[\[Beta] + \[Tau]]\ \[Omega]\_1\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Expanding the right-hand side of ",
  StyleBox["eq33c",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a Fourier series using trigonometric identities yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq33c[\([2]\)] = eq33c[\([2]\)] // TrigReduce\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(1\/2\ 
      \((\(-a\^2\)\ \[Alpha]\_2 - 
          a\^2\ Cos[2\ \[Beta] + 2\ \[Tau]]\ \[Alpha]\_2 + 
          4\ a\ Cos[\[Beta] + \[Tau]]\ \[Omega]\_1)\)\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Eliminating the terms, ",
  Cell[BoxData[
      \(TraditionalForm\`Cos[\[Beta] + \[Tau]]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`Sin[\[Beta] + \[Tau]]\)]],
  ", demands that ",
  Cell[BoxData[
      \(\[Omega]\_1 = 0\)]],
  ". Then, the particular solution of ",
  StyleBox["eq33c",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be expressed as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol2 = 
      \(DSolve[eq33c /. \[Omega]\_1 -> 0, u\_2[\[Tau]], \[Tau]]\)[\([1]\)] /. 
          C[_] -> 0 // Simplify\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_2[\[Tau]] \[Rule] 
        1\/6\ a\^2\ \((\(-3\) + Cos[2\ \((\[Beta] + \[Tau])\)])\)\ 
          \[Alpha]\_2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["or in a pure function format as", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol21 = u\_2 -> Function[\[Tau], sol2[\([1, 2]\)] // Evaluate]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u\_2 \[Rule] 
      Function[\[Tau], 
        1\/6\ a\^2\ \((\(-3\) + Cos[2\ \((\[Beta] + \[Tau])\)])\)\ 
          \[Alpha]\_2]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting ",
  StyleBox["sol1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["sol21",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the third-order equation, ",
  StyleBox["eqEps[[3]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and using the fact that ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_1 = 0\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"eq33d", "=", 
      RowBox[{
        RowBox[{
          RowBox[{\(u\_3[\[Tau]]\), "+", 
            RowBox[{
              SubsuperscriptBox["u", "3", "\[DoublePrime]",
                MultilineFunction->None], "[", "\[Tau]", "]"}], "-", "#"}], 
          " ", "&"}], " ", "/@", " ", 
        \(Reverse[
          \(\(eqEps[\([3]\)] /. \[Omega]\_1 -> 0\) /. sol1\) /. sol21 // 
            Simplify]\)}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(u\_3[\[Tau]]\), "+", 
        RowBox[{
          SubsuperscriptBox["u", "3", "\[DoublePrime]",
            MultilineFunction->None], "[", "\[Tau]", "]"}]}], "==", 
      \(\(-\(1\/3\)\)\ a\^3\ Cos[\[Beta] + \[Tau]]\ 
          \((\(-3\) + Cos[2\ \((\[Beta] + \[Tau])\)])\)\ \[Alpha]\_2\%2 - 
        a\^3\ Cos[\[Beta] + \[Tau]]\^3\ \[Alpha]\_3 + 
        2\ a\ Cos[\[Beta] + \[Tau]]\ \[Omega]\_2\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Expanding the right-hand side of ",
  StyleBox["eq33d",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a Fourier series using trigonometric identities, we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq33d[\([2]\)] = eq33d[\([2]\)] // TrigReduce\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(1\/12\ 
      \((10\ a\^3\ Cos[\[Beta] + \[Tau]]\ \[Alpha]\_2\%2 - 
          2\ a\^3\ Cos[3\ \[Beta] + 3\ \[Tau]]\ \[Alpha]\_2\%2 - 
          9\ a\^3\ Cos[\[Beta] + \[Tau]]\ \[Alpha]\_3 - 
          3\ a\^3\ Cos[3\ \[Beta] + 3\ \[Tau]]\ \[Alpha]\_3 + 
          24\ a\ Cos[\[Beta] + \[Tau]]\ \[Omega]\_2)\)\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Eliminating the terms that lead to secular terms from ",
  StyleBox["eq33d[[2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " demands that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(omg2Rule = 
      \(Solve[Coefficient[eq33d[\([2]\)], Cos[\[Tau] + \[Beta]]] == 0, 
            \[Omega]\_2]\)[\([1]\)] // ExpandAll\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Omega]\_2 \[Rule] 
        \(-\(5\/12\)\)\ a\^2\ \[Alpha]\_2\%2 + \(3\ a\^2\ \[Alpha]\_3\)\/8}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "As discussed above, for a second-order uniform expansion, we do not need \
to solve for ",
  Cell[BoxData[
      \(TraditionalForm\`u\_3[\[Tau]]\)]],
  ". Combining the first- and second-order solutions, we obtain, to the \
second approximation, that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(u[\[Tau]] == 
      \((\(\(\(u[\[Tau]] /. uExpRule[2]\) /. sol1\) /. sol21 // Simplify\) // 
          Expand)\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u[\[Tau]] == 
      a\ \[Epsilon]\ Cos[\[Beta] + \[Tau]] - 
        1\/2\ a\^2\ \[Epsilon]\^2\ \[Alpha]\_2 + 
        1\/6\ a\^2\ \[Epsilon]\^2\ Cos[2\ \((\[Beta] + \[Tau])\)]\ 
          \[Alpha]\_2 + \[Epsilon]\^3\ u\_3[\[Tau]]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["where", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(\(\(\[Tau] == \[Omega]\ t /. omgRule[2]\) /. omg2Rule\) /. 
      \[Omega]\_1 -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\[Tau] == 
      t\ \((1 + 
            \[Epsilon]\^2\ 
              \((\(-\(5\/12\)\)\ a\^2\ \[Alpha]\_2\%2 + 
                  \(3\ a\^2\ \[Alpha]\_3\)\/8)\))\)\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The above expansion is uniform to second order because secular terms do \
not appear in it and the correction term (the term proportional to ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^2\)]],
  ") is small compared with the first term."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Returning to ",
  StyleBox["sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in the previous section, we note that the first secular term appears at ",
  
  StyleBox["O",
    FontSlant->"Italic"],
  "(",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^3\)]],
  "). Consequently, we could have concluded that ",
  Cell[BoxData[
      \(TraditionalForm\`\[Omega]\_1 = 0\)]],
  " before carrying out the expansion because the term ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\ \[Omega]\_1\)]],
  " in ",
  StyleBox["omgRule[2]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " creates secular terms at ",
  Cell[BoxData[
      \(TraditionalForm\`O(\[Epsilon]\^2)\)]],
  " and not at ",
  Cell[BoxData[
      \(TraditionalForm\`O(\[Epsilon]\^3)\)]],
  ", as needed to eliminate the secular term from ",
  StyleBox["sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["3.4 The Method of Multiple Scales", "Section",
  CellMargins->{{Inherited, 17.4375}, {Inherited, Inherited}},
  Evaluatable->False,
  TextAlignment->Left,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],

Cell[TextData[{
  "We use the method of multiple scales to attack directly ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in Section 3.4.1, the corresponding first-order real-valued equations of \
",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in Section 3.4.2, and the corresponding first-order complex-valued \
equation of ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in Section 3.4.3. To obtain a second-order uniform expansion by using the \
method of multiple scales, we need the three time scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0 = t, T\_1 = \[Epsilon]\ t, \)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_2 = \[Epsilon]\^2\ t\)]],
  ", which we symbolize by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(Needs["\<Utilities`Notation`\>"]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"Symbolize", "[", 
        TagBox[\(T\_0\),
          NotationBoxTag,
          Editable->True], "]"}], ";", 
      RowBox[{"Symbolize", "[", 
        TagBox[\(T\_1\),
          NotationBoxTag,
          Editable->True], "]"}], ";", 
      RowBox[{"Symbolize", "[", 
        TagBox[\(T\_2\),
          NotationBoxTag,
          Editable->True], "]"}], ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["and list as", "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(timeScales = {T\_0, T\_1, T\_2}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In terms of the time scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_n\)]],
  ", the time derivatives become"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(\ \(dt[1]\)[expr_] := 
      Sum[\[Epsilon]\^i\ D[expr, timeScales[\([i + 1]\)]], {i, 0, maxOrder}]; 
    \n\ \(dt[2]\)[expr_] := 
      \((\(dt[1]\)[\(dt[1]\)[expr]] // Expand)\) /. 
        \[Epsilon]\^\(i_ /; i > maxOrder\) -> 0; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["For a uniform second-order expansion, we let", "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(maxOrder = 2; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
To represent some of the expressions in a more concise way, we introduce the \
following display rule:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(displayRule = {
        \(\(Derivative[a__]\)[u_\_i_]\)[__] :> 
          SequenceForm[
            Times\ @@\ MapIndexed[D\_\(#2[\([1]\)] - 1\)\%#1\ &, {a}], u\_i], 
        \(\(Derivative[a__]\)[A]\)[__] :> 
          SequenceForm[Times\ @@\ MapIndexed[D\_\(#2[\([1]\)]\)\%#1\ &, {a}], 
            A], \(\(Derivative[a__]\)[A\&_]\)[__] :> 
          SequenceForm[Times\ @@\ MapIndexed[D\_\(#2[\([1]\)]\)\%#1\ &, {a}], 
            A\&_], u_\_i_[__] -> u\_i, A[__] -> A, A\&_[__] -> A\&_}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell["3.4.1 Second-Order Real-Valued System", "Subsection",
  CellMargins->{{Inherited, 17.4375}, {Inherited, Inherited}},
  Evaluatable->False,
  TextAlignment->Left,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],

Cell[TextData[{
  "Using the derivative rule, we transform ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into the partial-differential equation"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq341a = 
      eq31g /. {u[t] -> u[T\_0, T\_1, T\_2], 
          \(\(Derivative[n_]\)[u]\)[t] -> \(dt[n]\)[u[T\_0, T\_1, T\_2]]}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(u[T\_0, T\_1, T\_2]\), "+", \(\[Alpha]\_2\ u[T\_0, T\_1, T\_2]\^2\), 
        "+", \(\[Alpha]\_3\ u[T\_0, T\_1, T\_2]\^3\), "+", 
        RowBox[{\(\[Epsilon]\^2\), " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((0, 2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
        "+", 
        RowBox[{"2", " ", \(\[Epsilon]\^2\), " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((1, 0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
        "+", 
        RowBox[{"2", " ", "\[Epsilon]", " ", 
          RowBox[{
            SuperscriptBox["u", 
              TagBox[\((1, 1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
        "+", 
        RowBox[{
          SuperscriptBox["u", 
            TagBox[\((2, 0, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
      "==", "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Again, to determine a uniform expansion of order maxOrder, we expand ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  " to order maxOrder+1, eliminate the secular terms from the maxOrder+1 \
equation, but do not include its solution in the final approximate solution. \
Hence, we seek a uniform expansion of the solution of ",
  StyleBox["eq341a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in the form"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(solRule = 
      u -> \((Sum[\[Epsilon]\^i\ u\_i[#1, #2, #3], {i, maxOrder + 1}]\ &)\); 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting this expansion into ",
  StyleBox["eq341a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and discarding terms of order \
higher than ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^3\)]],
  ", we have"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq341b = 
      \((eq341a /. solRule // ExpandAll)\) /. 
        \[Epsilon]\^\(n_ /; n > maxOrder + 1\) -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(\[Epsilon]\ u\_1[T\_0, T\_1, T\_2]\), "+", 
        \(\[Epsilon]\^2\ \[Alpha]\_2\ u\_1[T\_0, T\_1, T\_2]\^2\), "+", 
        \(\[Epsilon]\^3\ \[Alpha]\_3\ u\_1[T\_0, T\_1, T\_2]\^3\), "+", 
        \(\[Epsilon]\^2\ u\_2[T\_0, T\_1, T\_2]\), "+", 
        \(2\ \[Epsilon]\^3\ \[Alpha]\_2\ u\_1[T\_0, T\_1, T\_2]\ 
          u\_2[T\_0, T\_1, T\_2]\), "+", 
        \(\[Epsilon]\^3\ u\_3[T\_0, T\_1, T\_2]\), "+", 
        RowBox[{\(\[Epsilon]\^3\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "1", 
              TagBox[\((0, 2, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
        "+", 
        RowBox[{"2", " ", \(\[Epsilon]\^3\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "1", 
              TagBox[\((1, 0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
        "+", 
        RowBox[{"2", " ", \(\[Epsilon]\^2\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "1", 
              TagBox[\((1, 1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
        "+", 
        RowBox[{"2", " ", \(\[Epsilon]\^3\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "2", 
              TagBox[\((1, 1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
        "+", 
        RowBox[{"\[Epsilon]", " ", 
          RowBox[{
            SubsuperscriptBox["u", "1", 
              TagBox[\((2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
        "+", 
        RowBox[{\(\[Epsilon]\^2\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "2", 
              TagBox[\((2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
        "+", 
        RowBox[{\(\[Epsilon]\^3\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "3", 
              TagBox[\((2, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}]}], 
      "==", "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Equating coefficients of like powers of \[Epsilon] in ",
  StyleBox["eq341b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqEps = 
      \(CoefficientList[eq341b[\([1]\)], \[Epsilon]] == 0 // Thread\) // Rest
        \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(u\_1[T\_0, T\_1, T\_2]\), "+", 
            RowBox[{
              SubsuperscriptBox["u", "1", 
                TagBox[\((2, 0, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
          "==", "0"}], ",", 
        RowBox[{
          RowBox[{
          \(\[Alpha]\_2\ u\_1[T\_0, T\_1, T\_2]\^2\), "+", 
            \(u\_2[T\_0, T\_1, T\_2]\), "+", 
            RowBox[{"2", " ", 
              RowBox[{
                SubsuperscriptBox["u", "1", 
                  TagBox[\((1, 1, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), 
                "]"}]}], "+", 
            RowBox[{
              SubsuperscriptBox["u", "2", 
                TagBox[\((2, 0, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
          "==", "0"}], ",", 
        RowBox[{
          RowBox[{
          \(\[Alpha]\_3\ u\_1[T\_0, T\_1, T\_2]\^3\), "+", 
            \(2\ \[Alpha]\_2\ u\_1[T\_0, T\_1, T\_2]\ 
              u\_2[T\_0, T\_1, T\_2]\), "+", \(u\_3[T\_0, T\_1, T\_2]\), 
            "+", 
            RowBox[{
              SubsuperscriptBox["u", "1", 
                TagBox[\((0, 2, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}], 
            "+", 
            RowBox[{"2", " ", 
              RowBox[{
                SubsuperscriptBox["u", "1", 
                  TagBox[\((1, 0, 1)\),
                    Derivative],
                  MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), 
                "]"}]}], "+", 
            RowBox[{"2", " ", 
              RowBox[{
                SubsuperscriptBox["u", "2", 
                  TagBox[\((1, 1, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), 
                "]"}]}], "+", 
            RowBox[{
              SubsuperscriptBox["u", "3", 
                TagBox[\((2, 0, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
          "==", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
To place the linear operator on one side and the rest of the terms on the \
other, we define\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(eqOrder[i_] := 
      \(\((eqEps[\([1, 1]\)] /. u\_1 -> u\_i)\) - #\ &\)\ /@\ \ 
          eqEps[\([i]\)] // Reverse\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["eqOrder[i]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["displayRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we rewrite ",
  StyleBox["eqEps",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a concise way as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(Array[eqOrder, 3] /. displayRule // TableForm\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{
              RowBox[{
                InterpretationBox[\(D\_0\%2\[InvisibleSpace]u\_1\),
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Subscript[ u, 1]],
                  Editable->False], "+", \(u\_1\)}], "==", "0"}]},
          {
            RowBox[{
              RowBox[{
                InterpretationBox[\(D\_0\%2\[InvisibleSpace]u\_2\),
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Subscript[ u, 2]],
                  Editable->False], "+", \(u\_2\)}], "==", 
              RowBox[{
                RowBox[{\(-2\), " ", 
                  RowBox[{"(", 
                    
                    InterpretationBox[
                      \(\(D\_0\ D\_1\)\[InvisibleSpace]u\_1\),
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 1]], 
                        Subscript[ u, 1]],
                      Editable->False], ")"}]}], "-", 
                \(u\_1\%2\ \[Alpha]\_2\)}]}]},
          {
            RowBox[{
              RowBox[{
                InterpretationBox[\(D\_0\%2\[InvisibleSpace]u\_3\),
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 0], 2], 
                    Subscript[ u, 3]],
                  Editable->False], "+", \(u\_3\)}], "==", 
              RowBox[{
                RowBox[{\(-2\), " ", 
                  RowBox[{"(", 
                    
                    InterpretationBox[
                      \(\(D\_0\ D\_1\)\[InvisibleSpace]u\_2\),
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 1]], 
                        Subscript[ u, 2]],
                      Editable->False], ")"}]}], "-", 
                InterpretationBox[\(D\_1\%2\[InvisibleSpace]u\_1\),
                  SequenceForm[ 
                    Power[ 
                      Subscript[ D, 1], 2], 
                    Subscript[ u, 1]],
                  Editable->False], "-", 
                RowBox[{"2", " ", 
                  RowBox[{"(", 
                    
                    InterpretationBox[
                      \(\(D\_0\ D\_2\)\[InvisibleSpace]u\_1\),
                      SequenceForm[ 
                        Times[ 
                          Subscript[ D, 0], 
                          Subscript[ D, 2]], 
                        Subscript[ u, 1]],
                      Editable->False], ")"}]}], "-", 
                \(2\ u\_1\ u\_2\ \[Alpha]\_2\), "-", 
                \(u\_1\%3\ \[Alpha]\_3\)}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Plus[ 
            SequenceForm[ 
              Power[ 
                Subscript[ D, 0], 2], 
              Subscript[ u, 1]], 
            Subscript[ u, 1]], 0], 
        Equal[ 
          Plus[ 
            SequenceForm[ 
              Power[ 
                Subscript[ D, 0], 2], 
              Subscript[ u, 2]], 
            Subscript[ u, 2]], 
          Plus[ 
            Times[ -2, 
              SequenceForm[ 
                Times[ 
                  Subscript[ D, 0], 
                  Subscript[ D, 1]], 
                Subscript[ u, 1]]], 
            Times[ -1, 
              Power[ 
                Subscript[ u, 1], 2], 
              Subscript[ \[Alpha], 2]]]], 
        Equal[ 
          Plus[ 
            SequenceForm[ 
              Power[ 
                Subscript[ D, 0], 2], 
              Subscript[ u, 3]], 
            Subscript[ u, 3]], 
          Plus[ 
            Times[ -2, 
              SequenceForm[ 
                Times[ 
                  Subscript[ D, 0], 
                  Subscript[ D, 1]], 
                Subscript[ u, 2]]], 
            Times[ -1, 
              SequenceForm[ 
                Power[ 
                  Subscript[ D, 1], 2], 
                Subscript[ u, 1]]], 
            Times[ -2, 
              SequenceForm[ 
                Times[ 
                  Subscript[ D, 0], 
                  Subscript[ D, 2]], 
                Subscript[ u, 1]]], 
            Times[ -2, 
              Subscript[ u, 1], 
              Subscript[ u, 2], 
              Subscript[ \[Alpha], 2]], 
            Times[ -1, 
              Power[ 
                Subscript[ u, 1], 3], 
              Subscript[ \[Alpha], 3]]]]}]]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm
      \`D\_n\ u\_i = \[PartialD]u\_i[T\_0, T\_1, T\_2]/\[PartialD]T\_n\)]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "The general solution of the first-order equation, ",
  StyleBox["eqOrder[1]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", can be expressed as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(sol1 = {
        u\_1 -> \((A[#2, #3]\ Exp[I\ #1] + A\&_[#2, #3]\ Exp[\(-I\)\ #1]\ &)
            \)}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`A\&_\)]],
  " is the complex conjugate of ",
  Cell[BoxData[
      \(TraditionalForm\`A\)]],
  " defined by the conjugate rule"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(conjugateRule = {A -> A\&_, A\&_ -> A, 
        Complex[0, n_] -> Complex[0, \(-n\)]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Then, the second-order equation, ",
  StyleBox["eqOrder[2]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", becomes"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq341c = eqOrder[2] /. sol1 // ExpandAll; \neq341c /. displayRule\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        InterpretationBox[\(D\_0\%2\[InvisibleSpace]u\_2\),
          SequenceForm[ 
            Power[ 
              Subscript[ D, 0], 2], 
            Subscript[ u, 2]],
          Editable->False], "+", \(u\_2\)}], "==", 
      RowBox[{
        RowBox[{\(-2\), " ", "I", " ", \(E\^\(I\ T\_0\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\),
              SequenceForm[ 
                Subscript[ D, 1], A],
              Editable->False], ")"}]}], "+", 
        RowBox[{"2", " ", "I", " ", \(E\^\(\(-I\)\ T\_0\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\),
              SequenceForm[ 
                Subscript[ D, 1], 
                OverBar[ A]],
              Editable->False], ")"}]}], "-", 
        \(A\^2\ E\^\(2\ I\ T\_0\)\ \[Alpha]\_2\), "-", 
        \(2\ A\ A\&_\ \[Alpha]\_2\), "-", 
        \(E\^\(\(-2\)\ I\ T\_0\)\ A\&_\^2\ \[Alpha]\_2\)}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Eliminating the term ",
  Cell[BoxData[
      \(TraditionalForm\`E\^\(I\ T\_0\)\)]],
  " that produces secular terms in ",
  Cell[BoxData[
      \(TraditionalForm\`u\_2\)]],
  " from the right-hand side of ",
  StyleBox["eq341c",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " demands that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond1 = Coefficient[eq341c[\([2]\)], Exp[I\ T\_0]] == 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(-2\), " ", "I", " ", 
        RowBox[{
          SuperscriptBox["A", 
            TagBox[\((1, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["or", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{
      RowBox[{"SCond1Rule", "=", 
        RowBox[{
          RowBox[{"Solve", "[", 
            RowBox[{"SCond1", ",", 
              RowBox[{
                SuperscriptBox["A", 
                  TagBox[\((1, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], 
            "]"}], "[", \([1]\), "]"}]}], "\n", 
      RowBox[{"(*", " ", 
        RowBox[{"either", " ", "copy", " ", "and", " ", "paste", " ", 
          RowBox[{
            SuperscriptBox["A", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], " ", 
          "from", " ", "SCond1", " ", "or", " ", "use", " ", 
          \(\(\(Derivative[1, 0]\)[A]\)[T\_1, T\_2]\), " ", "as", " ", "the", 
          " ", "second", " ", "argument", " ", "to", " ", "Solve"}], " ", 
        "*)"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          SuperscriptBox["A", 
            TagBox[\((1, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], "\[Rule]", 
        "0"}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Eliminating the term ",
  Cell[BoxData[
      \(TraditionalForm\`E\^\(\(-I\)\ T\_0\)\)]],
  " that produces secular terms in ",
  Cell[BoxData[
      \(TraditionalForm\`u\_2\)]],
  " from the right-hand side of ",
  StyleBox["eq341c",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields the complex conjugate of ",
  StyleBox["SCond1Rule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "; that is,"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ccSCond1Rule = SCond1Rule /. conjugateRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          SuperscriptBox[\(A\&_\), 
            TagBox[\((1, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], "\[Rule]", 
        "0"}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "It follows from ",
  StyleBox["SCond1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that ",
  Cell[BoxData[
      \(TraditionalForm\`A = A[T\_2]\)]],
  ". Substituting the solvability conditions into ",
  StyleBox["eq341c",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq341d = \(eq341c /. SCond1Rule\) /. ccSCond1Rule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(u\_2[T\_0, T\_1, T\_2]\), "+", 
        RowBox[{
          SubsuperscriptBox["u", "2", 
            TagBox[\((2, 0, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
      "==", \(\(-E\^\(2\ I\ T\_0\)\)\ A[T\_1, T\_2]\^2\ \[Alpha]\_2 - 
        2\ A[T\_1, T\_2]\ \[Alpha]\_2\ A\&_[T\_1, T\_2] - 
        E\^\(\(-2\)\ I\ T\_0\)\ \[Alpha]\_2\ A\&_[T\_1, T\_2]\^2\)}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The particular solution of ",
  StyleBox["eq341d",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be obtained by using ",
  StyleBox["DSolve",
    FontWeight->"Bold"],
  "; the result is"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(u2Sol = 
      \(\(DSolve[eq341d, u\_2[T\_0, T\_1, T\_2], timeScales]\)[\([1]\)] /. 
            C[_] -> \((0\ &)\) // TrigToExp\) // ExpandAll\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_2[T\_0, T\_1, T\_2] \[Rule] 
        1\/3\ E\^\(2\ I\ T\_0\)\ A[T\_1, T\_2]\^2\ \[Alpha]\_2 - 
          2\ A[T\_1, T\_2]\ \[Alpha]\_2\ A\&_[T\_1, T\_2] + 
          1\/3\ E\^\(\(-2\)\ I\ T\_0\)\ \[Alpha]\_2\ A\&_[T\_1, T\_2]\^2}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "whose right-hand side can be used directly to express ",
  Cell[BoxData[
      \(TraditionalForm\`u\_2\)]],
  " in a pure function form as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol2 = {
        u\_2 -> Function[{T\_0, T\_1, T\_2}, u2Sol[\([1, 2]\)] // Evaluate]}
        \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_2 \[Rule] 
        Function[{T\_0, T\_1, T\_2}, 
          1\/3\ E\^\(2\ I\ T\_0\)\ A[T\_1, T\_2]\^2\ \[Alpha]\_2 - 
            2\ A[T\_1, T\_2]\ \[Alpha]\_2\ A\&_[T\_1, T\_2] + 
            1\/3\ E\^\(\(-2\)\ I\ T\_0\)\ \[Alpha]\_2\ A\&_[T\_1, T\_2]\^2]}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting the first- and second-order solutions, ",
  StyleBox["sol1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["sol2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", into the third-order equation, ",
  StyleBox["eqOrder[3]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq341e = \(eqOrder[3] /. sol1\) /. sol2 // ExpandAll; \n
    eq341e /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        InterpretationBox[\(D\_0\%2\[InvisibleSpace]u\_3\),
          SequenceForm[ 
            Power[ 
              Subscript[ D, 0], 2], 
            Subscript[ u, 3]],
          Editable->False], "+", \(u\_3\)}], "==", 
      RowBox[{
        RowBox[{\(-E\^\(I\ T\_0\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\%2\[InvisibleSpace]A\),
              SequenceForm[ 
                Power[ 
                  Subscript[ D, 1], 2], A],
              Editable->False], ")"}]}], "-", 
        RowBox[{\(E\^\(\(-I\)\ T\_0\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\%2\[InvisibleSpace]A\&_\),
              SequenceForm[ 
                Power[ 
                  Subscript[ D, 1], 2], 
                OverBar[ A]],
              Editable->False], ")"}]}], "-", 
        RowBox[{"2", " ", "I", " ", \(E\^\(I\ T\_0\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_2\[InvisibleSpace]A\),
              SequenceForm[ 
                Subscript[ D, 2], A],
              Editable->False], ")"}]}], "+", 
        RowBox[{"2", " ", "I", " ", \(E\^\(\(-I\)\ T\_0\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_2\[InvisibleSpace]A\&_\),
              SequenceForm[ 
                Subscript[ D, 2], 
                OverBar[ A]],
              Editable->False], ")"}]}], "-", 
        RowBox[{
        \(8\/3\), " ", "I", " ", "A", " ", \(E\^\(2\ I\ T\_0\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\),
              SequenceForm[ 
                Subscript[ D, 1], A],
              Editable->False], ")"}], " ", \(\[Alpha]\_2\)}], "+", 
        RowBox[{
        \(8\/3\), " ", "I", " ", \(E\^\(\(-2\)\ I\ T\_0\)\), " ", \(A\&_\), 
          " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\),
              SequenceForm[ 
                Subscript[ D, 1], 
                OverBar[ A]],
              Editable->False], ")"}], " ", \(\[Alpha]\_2\)}], "-", 
        \(2\/3\ A\^3\ E\^\(3\ I\ T\_0\)\ \[Alpha]\_2\%2\), "+", 
        \(10\/3\ A\^2\ E\^\(I\ T\_0\)\ A\&_\ \[Alpha]\_2\%2\), "+", 
        \(10\/3\ A\ E\^\(\(-I\)\ T\_0\)\ A\&_\^2\ \[Alpha]\_2\%2\), "-", 
        \(2\/3\ E\^\(\(-3\)\ I\ T\_0\)\ A\&_\^3\ \[Alpha]\_2\%2\), "-", 
        \(A\^3\ E\^\(3\ I\ T\_0\)\ \[Alpha]\_3\), "-", 
        \(3\ A\^2\ E\^\(I\ T\_0\)\ A\&_\ \[Alpha]\_3\), "-", 
        \(3\ A\ E\^\(\(-I\)\ T\_0\)\ A\&_\^2\ \[Alpha]\_3\), "-", 
        \(E\^\(\(-3\)\ I\ T\_0\)\ A\&_\^3\ \[Alpha]\_3\)}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Eliminating the terms that produce secular terms in ",
  Cell[BoxData[
      \(TraditionalForm\`u\_3\)]],
  " from the right-hand side of ",
  StyleBox["eq341e",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " demands that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond2 = Coefficient[eq341e[\([2]\)], Exp[I\ T\_0]] == 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(10\/3\ A[T\_1, T\_2]\^2\ \[Alpha]\_2\%2\ A\&_[T\_1, T\_2]\), "-", 
        \(3\ A[T\_1, T\_2]\^2\ \[Alpha]\_3\ A\&_[T\_1, T\_2]\), "-", 
        RowBox[{"2", " ", "I", " ", 
          RowBox[{
            SuperscriptBox["A", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], "-", 
        RowBox[{
          SuperscriptBox["A", 
            TagBox[\((2, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Using ",
  StyleBox["SCond1Rule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[" ",
    FontSlant->"Italic"],
  "and the fact that ",
  Cell[BoxData[
      \(TraditionalForm\`A = A[T\_2]\)]],
  ", we can rewrite ",
  StyleBox["SCond2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond = 
      \(SCond2 /. D[SCond1Rule, T\_1]\) /. {A -> \((A[#2]\ &)\), 
          A\&_ -> \((A\&_[#2]\ &)\)}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(10\/3\ A[T\_2]\^2\ \[Alpha]\_2\%2\ A\&_[T\_2]\), "-", 
        \(3\ A[T\_2]\^2\ \[Alpha]\_3\ A\&_[T\_2]\), "-", 
        RowBox[{"2", " ", "I", " ", 
          RowBox[{
            SuperscriptBox["A", "\[Prime]",
              MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Expressing ",
  Cell[BoxData[
      \(TraditionalForm\`A\)]],
  " in the polar form"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ruleA = {A -> \((\(1\/2\) a[#]\ Exp[I\ \[Beta][#]]\ &)\), 
        A\&_ -> \((\(1\/2\) a[#]\ Exp[\(-I\)\ \[Beta][#]]\ &)\)}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  StyleBox["a",
    FontSlant->"Italic"],
  " and \[Beta] are real and using the rule"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(realRule = {Re[s_] -> s, Im[s_] -> 0}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "we write ",
  StyleBox["SCond",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq341f = 
      SCond[\([1]\)]\ Exp[\(-I\)\ \[Beta][T\_2]] /. ruleA // Expand\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
    \(5\/12\ a[T\_2]\^3\ \[Alpha]\_2\%2\), "-", 
      \(3\/8\ a[T\_2]\^3\ \[Alpha]\_3\), "-", 
      RowBox[{"I", " ", 
        RowBox[{
          SuperscriptBox["a", "\[Prime]",
            MultilineFunction->None], "[", \(T\_2\), "]"}]}], "+", 
      RowBox[{\(a[T\_2]\), " ", 
        RowBox[{
          SuperscriptBox["\[Beta]", "\[Prime]",
            MultilineFunction->None], "[", \(T\_2\), "]"}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Separating the real and imaginary parts in ",
  StyleBox["eq341f",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the modulation equations"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ampEq = Im[eq341f] == 0 /. realRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"-", 
        RowBox[{
          SuperscriptBox["a", "\[Prime]",
            MultilineFunction->None], "[", \(T\_2\), "]"}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phaseEq = Re[eq341f] == 0 /. realRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(5\/12\ a[T\_2]\^3\ \[Alpha]\_2\%2\), "-", 
        \(3\/8\ a[T\_2]\^3\ \[Alpha]\_3\), "+", 
        RowBox[{\(a[T\_2]\), " ", 
          RowBox[{
            SuperscriptBox["\[Beta]", "\[Prime]",
              MultilineFunction->None], "[", \(T\_2\), "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The solution of ",
  StyleBox["ampEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " is ",
  Cell[BoxData[
      \(TraditionalForm\`a = \(a\_0 = const\)\)]],
  ". Then, if ",
  Cell[BoxData[
      \(TraditionalForm\`a\_0 \[NotEqual] 0\)]],
  ", the solution of ",
  StyleBox["phaseEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " is"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(betaRule = 
      \(DSolve[phaseEq /. a[T\_2] -> a\_0, \[Beta][T\_2], T\_2]\)[\([1]\)] /. 
          C[1] -> \[Beta]\_0 // ExpandAll\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Beta][T\_2] \[Rule] 
        \(-\(5\/12\)\)\ T\_2\ a\_0\%2\ \[Alpha]\_2\%2 + 
          3\/8\ T\_2\ a\_0\%2\ \[Alpha]\_3 + \[Beta]\_0}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`\[Beta]\_0\)]],
  " is a constant. Substituting ",
  StyleBox["ruleA",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["sol1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["sol2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and recalling that ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0 = t\)]],
  StyleBox[" ",
    FontSlant->"Italic"],
  "and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_2 = \[Epsilon]\^2\ t\)]],
  ", we obtain, to the second approximation, that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(u[t] == 
      \((\(\(\(\((
                  \(\(\(\(\(u[T\_0, T\_1, T\_2] /. solRule\) /. sol1\) /. sol2
                            \) /. \[Epsilon]\^3 -> 0\) /. {A[__] -> A[T\_2], 
                          A\&_[__] -> A\&_[T\_2]}\) /. ruleA // ExpToTrig)
                  \) /. {a[T\_2] -> a\_0}\) /. betaRule\) /. {T\_0 -> t, 
              T\_2 -> \[Epsilon]\^2\ t}\) /. 
          Cos[arg_] :> Cos[Collect[arg, t]])\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u[t] == 
      \[Epsilon]\ 
          Cos[t\ \((
                  1 - 5\/12\ \[Epsilon]\^2\ a\_0\%2\ \[Alpha]\_2\%2 + 
                    3\/8\ \[Epsilon]\^2\ a\_0\%2\ \[Alpha]\_3)\) + \[Beta]\_0]
            \ a\_0 - 1\/2\ \[Epsilon]\^2\ a\_0\%2\ \[Alpha]\_2 + 
        1\/6\ \[Epsilon]\^2\ 
          Cos[t\ \((
                  2 - 5\/6\ \[Epsilon]\^2\ a\_0\%2\ \[Alpha]\_2\%2 + 
                    3\/4\ \[Epsilon]\^2\ a\_0\%2\ \[Alpha]\_3)\) + 
              2\ \[Beta]\_0]\ a\_0\%2\ \[Alpha]\_2\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
which is in full agreement with that obtained by using the Lindstedt-Poincar\
\[EAcute] technique.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["3.4.2 First-Order Real-Valued System", "Subsection",
  CellMargins->{{Inherited, 17.4375}, {Inherited, Inherited}},
  Evaluatable->False,
  TextAlignment->Left,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],

Cell[TextData[{
  "In this section, we first transform ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into a system of two real-valued first-order equations using the \
transformation ",
  Cell[BoxData[
      \(TraditionalForm\`v[t] = \(u\^\[Prime]\)[t]\)]],
  " and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq342a = {\(u\^\[Prime]\)[t] == v[t], 
        eq31g /. \(u\^\[DoublePrime]\)[t] -> \(v\^\[Prime]\)[t]}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox["u", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], "==", \(v[t]\)}], 
        ",", 
        RowBox[{
          RowBox[{
          \(u[t]\), "+", \(\[Alpha]\_2\ u[t]\^2\), "+", 
            \(\[Alpha]\_3\ u[t]\^3\), "+", 
            RowBox[{
              SuperscriptBox["v", "\[Prime]",
                MultilineFunction->None], "[", "t", "]"}]}], "==", "0"}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine a second-order uniform expansion of the solution of ",
  StyleBox["eq342a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " using the method of multiple scales, we first introduce"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"multiScales", "=", 
        RowBox[{"{", 
          RowBox[{
          \(u[t] -> u[T\_0, T\_1, T\_2]\), ",", 
            \(v[t] -> v[T\_0, T\_1, T\_2]\), ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox["u", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}], "->", 
              \(\(dt[1]\)[u[T\_0, T\_1, T\_2]]\)}], ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox["v", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}], "->", 
              \(\(dt[1]\)[v[T\_0, T\_1, T\_2]]\)}]}], "}"}]}], ";"}]], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "to transform the derivative with respect to ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  " in terms of the derivatives with respect to the three time scales ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0, T\_1\)]],
  ", and ",
  Cell[BoxData[
      \(TraditionalForm\`T\_2\)]],
  " and then expand ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`v\)]],
  " in the form"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(solRule = {
        u -> \((Sum[\[Epsilon]\^j\ u\_j[#1, #2, #3], {j, 3}]\ &)\), 
        v -> \((Sum[\[Epsilon]\^j\ v\_j[#1, #2, #3], {j, 3}]\ &)\)}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["multiScales",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["solRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eq342a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and discarding terms of order \
higher than ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^3\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eq342b = 
      \((\(eq342a /. multiScales\) /. solRule // ExpandAll)\) /. 
        \[Epsilon]\^\(n_ /; n > 3\) -> 0; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Equating coefficients of like powers of \[Epsilon] in ",
  StyleBox["eq342b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eqEps = 
      \(\(Thread[CoefficientList[Subtract\ @@\ #, \[Epsilon]] == 0]\ &\)\ /@\ 
            eq342b // Transpose\) // Rest; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
To place the linear operator on one side and the rest of the terms on the \
other, we define\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(eqOrder[i_] := 
      \((\(#[\([1]\)]\ &\)\ /@\ eqEps[\([1]\)] /. u_\_1 -> u\_i)\) == 
          \((\(#[\([1]\)]\ &\)\ /@\ eqEps[\([1]\)] /. u_\_1 -> u\_i)\) - 
            \((\(#[\([1]\)]\ &\)\ /@\ eqEps[\([i]\)])\) // Thread\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["eqOrder[i]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["displayRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we rewrite ",
  StyleBox["eqEps",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a concise way as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[{
    \(eqOrder[1] /. displayRule // TableForm\), 
    \(eqOrder[2] /. displayRule // TableForm\), 
    \(eqOrder[3] /. displayRule // TableForm\)}], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{
              RowBox[{
                InterpretationBox[\(D\_0\[InvisibleSpace]u\_1\),
                  SequenceForm[ 
                    Subscript[ D, 0], 
                    Subscript[ u, 1]],
                  Editable->False], "-", \(v\_1\)}], "==", "0"}]},
          {
            RowBox[{
              RowBox[{
                InterpretationBox[\(D\_0\[InvisibleSpace]v\_1\),
                  SequenceForm[ 
                    Subscript[ D, 0], 
                    Subscript[ v, 1]],
                  Editable->False], "+", \(u\_1\)}], "==", "0"}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Plus[ 
            SequenceForm[ 
              Subscript[ D, 0], 
              Subscript[ u, 1]], 
            Times[ -1, 
              Subscript[ v, 1]]], 0], 
        Equal[ 
          Plus[ 
            SequenceForm[ 
              Subscript[ D, 0], 
              Subscript[ v, 1]], 
            Subscript[ u, 1]], 0]}]]], "Output",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{
              RowBox[{
                InterpretationBox[\(D\_0\[InvisibleSpace]u\_2\),
                  SequenceForm[ 
                    Subscript[ D, 0], 
                    Subscript[ u, 2]],
                  Editable->False], "-", \(v\_2\)}], "==", 
              RowBox[{"-", 
                RowBox[{"(", 
                  InterpretationBox[\(D\_1\[InvisibleSpace]u\_1\),
                    SequenceForm[ 
                      Subscript[ D, 1], 
                      Subscript[ u, 1]],
                    Editable->False], ")"}]}]}]},
          {
            RowBox[{
              RowBox[{
                InterpretationBox[\(D\_0\[InvisibleSpace]v\_2\),
                  SequenceForm[ 
                    Subscript[ D, 0], 
                    Subscript[ v, 2]],
                  Editable->False], "+", \(u\_2\)}], "==", 
              RowBox[{
                RowBox[{"-", 
                  RowBox[{"(", 
                    InterpretationBox[\(D\_1\[InvisibleSpace]v\_1\),
                      SequenceForm[ 
                        Subscript[ D, 1], 
                        Subscript[ v, 1]],
                      Editable->False], ")"}]}], "-", 
                \(u\_1\%2\ \[Alpha]\_2\)}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Plus[ 
            SequenceForm[ 
              Subscript[ D, 0], 
              Subscript[ u, 2]], 
            Times[ -1, 
              Subscript[ v, 2]]], 
          Times[ -1, 
            SequenceForm[ 
              Subscript[ D, 1], 
              Subscript[ u, 1]]]], 
        Equal[ 
          Plus[ 
            SequenceForm[ 
              Subscript[ D, 0], 
              Subscript[ v, 2]], 
            Subscript[ u, 2]], 
          Plus[ 
            Times[ -1, 
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ v, 1]]], 
            Times[ -1, 
              Power[ 
                Subscript[ u, 1], 2], 
              Subscript[ \[Alpha], 2]]]]}]]], "Output",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{
              RowBox[{
                InterpretationBox[\(D\_0\[InvisibleSpace]u\_3\),
                  SequenceForm[ 
                    Subscript[ D, 0], 
                    Subscript[ u, 3]],
                  Editable->False], "-", \(v\_3\)}], "==", 
              RowBox[{
                RowBox[{"-", 
                  RowBox[{"(", 
                    InterpretationBox[\(D\_1\[InvisibleSpace]u\_2\),
                      SequenceForm[ 
                        Subscript[ D, 1], 
                        Subscript[ u, 2]],
                      Editable->False], ")"}]}], "-", 
                InterpretationBox[\(D\_2\[InvisibleSpace]u\_1\),
                  SequenceForm[ 
                    Subscript[ D, 2], 
                    Subscript[ u, 1]],
                  Editable->False]}]}]},
          {
            RowBox[{
              RowBox[{
                InterpretationBox[\(D\_0\[InvisibleSpace]v\_3\),
                  SequenceForm[ 
                    Subscript[ D, 0], 
                    Subscript[ v, 3]],
                  Editable->False], "+", \(u\_3\)}], "==", 
              RowBox[{
                RowBox[{"-", 
                  RowBox[{"(", 
                    InterpretationBox[\(D\_1\[InvisibleSpace]v\_2\),
                      SequenceForm[ 
                        Subscript[ D, 1], 
                        Subscript[ v, 2]],
                      Editable->False], ")"}]}], "-", 
                InterpretationBox[\(D\_2\[InvisibleSpace]v\_1\),
                  SequenceForm[ 
                    Subscript[ D, 2], 
                    Subscript[ v, 1]],
                  Editable->False], "-", \(2\ u\_1\ u\_2\ \[Alpha]\_2\), "-", 
                \(u\_1\%3\ \[Alpha]\_3\)}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Plus[ 
            SequenceForm[ 
              Subscript[ D, 0], 
              Subscript[ u, 3]], 
            Times[ -1, 
              Subscript[ v, 3]]], 
          Plus[ 
            Times[ -1, 
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ u, 2]]], 
            Times[ -1, 
              SequenceForm[ 
                Subscript[ D, 2], 
                Subscript[ u, 1]]]]], 
        Equal[ 
          Plus[ 
            SequenceForm[ 
              Subscript[ D, 0], 
              Subscript[ v, 3]], 
            Subscript[ u, 3]], 
          Plus[ 
            Times[ -1, 
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ v, 2]]], 
            Times[ -1, 
              SequenceForm[ 
                Subscript[ D, 2], 
                Subscript[ v, 1]]], 
            Times[ -2, 
              Subscript[ u, 1], 
              Subscript[ u, 2], 
              Subscript[ \[Alpha], 2]], 
            Times[ -1, 
              Power[ 
                Subscript[ u, 1], 3], 
              Subscript[ \[Alpha], 3]]]]}]]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["First-Order Equations: Linear System", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To determine the solution of the first-order equations, ",
  StyleBox["eqOrder[1]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we list their left-hand sides and obtain"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(linearSys = \(#[\([1]\)]\ &\)\ /@\ eqOrder[1]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{\(-v\_1[T\_0, T\_1, T\_2]\), "+", 
          RowBox[{
            SubsuperscriptBox["u", "1", 
              TagBox[\((1, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}], 
        ",", 
        RowBox[{\(u\_1[T\_0, T\_1, T\_2]\), "+", 
          RowBox[{
            SubsuperscriptBox["v", "1", 
              TagBox[\((1, 0, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), "]"}]}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Next, we seek a solution of ",
  StyleBox["linearSys",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in the form ",
  Cell[BoxData[
      \(TraditionalForm\`u\_1 = P\ E\^\(I\ T\_0\)\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`v\_1 = Q\ E\^\(I\ T\_0\)\)]],
  " and obtain"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(coefList = 
      E\^\(\(-I\)\ T\_0\)\ linearSys /. {u\_1 -> \((P\ E\^\(I\ #\)&)\), 
            v\_1 -> \((Q\ E\^\(I\ #\)&)\)} // Expand\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({I\ P - Q, P + I\ Q}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The coefficient matrix of ",
  StyleBox["coefList",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " is"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(coefMat = Outer[Coefficient, coefList, {P, Q}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({{I, \(-1\)}, {1, I}}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["and its adjoint is defined by", "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(hermitian[mat_?MatrixQ] := mat /. conjugateRule // Transpose\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Hence, the right and left eigenvectors of ",
  StyleBox["coefMat",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " are"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(rightVec = {1, c1} /. 
        \(Solve[\((coefMat . {1, c1})\)[\([1]\)] == 0, c1]\)[\([1]\)]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({1, I}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(leftVec = {1, c1} /. 
        \(Solve[\((hermitian[coefMat] . {1, c1})\)[\([1]\)] == 0, c1]\)[
          \([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({1, I}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Then the solution of the first-order equations can be expressed as\
\>", "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol1Form = rightVec\ A[T\_1, T\_2]\ E\^\(I\ T\_0\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({E\^\(I\ T\_0\)\ A[T\_1, T\_2], I\ E\^\(I\ T\_0\)\ A[T\_1, T\_2]}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "and hence ",
  Cell[BoxData[
      \(TraditionalForm\`u\_1\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`v\_1\)]],
  " can be expressed in a pure function form as"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order1Sol = {
        u\_1 -> Function[{T\_0, T\_1, T\_2}, 
            sol1Form[\([1]\)] + \((sol1Form[\([1]\)] /. conjugateRule)\) // 
              Evaluate], 
        v\_1 -> Function[{T\_0, T\_1, T\_2}, 
            sol1Form[\([2]\)] + \((sol1Form[\([2]\)] /. conjugateRule)\) // 
              Evaluate]}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_1 \[Rule] 
        Function[{T\_0, T\_1, T\_2}, 
          E\^\(I\ T\_0\)\ A[T\_1, T\_2] + 
            E\^\(\(-I\)\ T\_0\)\ A\&_[T\_1, T\_2]], 
      v\_1 \[Rule] 
        Function[{T\_0, T\_1, T\_2}, 
          I\ E\^\(I\ T\_0\)\ A[T\_1, T\_2] - 
            I\ E\^\(\(-I\)\ T\_0\)\ A\&_[T\_1, T\_2]]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Second-Order Equations", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the first-order solution, ",
  StyleBox["order1Sol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", into the second-order equations, ",
  StyleBox["eqOrder[2]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2Eq = eqOrder[2] /. order1Sol // ExpandAll; \n
    order2Eq /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            InterpretationBox[\(D\_0\[InvisibleSpace]u\_2\),
              SequenceForm[ 
                Subscript[ D, 0], 
                Subscript[ u, 2]],
              Editable->False], "-", \(v\_2\)}], "==", 
          RowBox[{
            RowBox[{\(-E\^\(I\ T\_0\)\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\),
                  SequenceForm[ 
                    Subscript[ D, 1], A],
                  Editable->False], ")"}]}], "-", 
            RowBox[{\(E\^\(\(-I\)\ T\_0\)\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    OverBar[ A]],
                  Editable->False], ")"}]}]}]}], ",", 
        RowBox[{
          RowBox[{
            InterpretationBox[\(D\_0\[InvisibleSpace]v\_2\),
              SequenceForm[ 
                Subscript[ D, 0], 
                Subscript[ v, 2]],
              Editable->False], "+", \(u\_2\)}], "==", 
          RowBox[{
            RowBox[{\(-I\), " ", \(E\^\(I\ T\_0\)\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\),
                  SequenceForm[ 
                    Subscript[ D, 1], A],
                  Editable->False], ")"}]}], "+", 
            RowBox[{"I", " ", \(E\^\(\(-I\)\ T\_0\)\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    OverBar[ A]],
                  Editable->False], ")"}]}], "-", 
            \(A\^2\ E\^\(2\ I\ T\_0\)\ \[Alpha]\_2\), "-", 
            \(2\ A\ A\&_\ \[Alpha]\_2\), "-", 
            \(E\^\(\(-2\)\ I\ T\_0\)\ A\&_\^2\ \[Alpha]\_2\)}]}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To eliminate the terms that produce secular terms in ",
  Cell[BoxData[
      \(TraditionalForm\`u\_2\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`v\_2\)]],
  ", we determine the vector proportional to ",
  Cell[BoxData[
      \(TraditionalForm\`E\^\(I\ T\_0\)\)]],
  " in the right-hand sides of ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(STerms1 = 
      \(Coefficient[#[\([2]\)], Exp[I\ T\_0]]\ &\)\ /@\ order2Eq\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"-", 
          RowBox[{
            SuperscriptBox["A", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], ",", 
        RowBox[{\(-I\), " ", 
          RowBox[{
            SuperscriptBox["A", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Then, the condition for the elimination of the terms that produce secular \
terms, solvability condition, demands that ",
  StyleBox["STerms1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " be orthogonal to the adjoint, left eigenvector. The result is"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"SCond1", "=", 
      RowBox[{
        RowBox[{"Solve", "[", 
          RowBox[{\(Conjugate[leftVec] . STerms1 == 0\), ",", 
            RowBox[{
              SuperscriptBox["A", 
                TagBox[\((1, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], 
          "]"}], "[", \([1]\), "]"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          SuperscriptBox["A", 
            TagBox[\((1, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], "\[Rule]", 
        "0"}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["whose complex conjugate is", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ccSCond1 = SCond1 /. conjugateRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          SuperscriptBox[\(A\&_\), 
            TagBox[\((1, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], "\[Rule]", 
        "0"}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "With this solvability condition, ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " becomes"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2Eqm = \(order2Eq /. SCond1\) /. ccSCond1; \n
    order2Eqm /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            InterpretationBox[\(D\_0\[InvisibleSpace]u\_2\),
              SequenceForm[ 
                Subscript[ D, 0], 
                Subscript[ u, 2]],
              Editable->False], "-", \(v\_2\)}], "==", "0"}], ",", 
        RowBox[{
          RowBox[{
            InterpretationBox[\(D\_0\[InvisibleSpace]v\_2\),
              SequenceForm[ 
                Subscript[ D, 0], 
                Subscript[ v, 2]],
              Editable->False], "+", \(u\_2\)}], "==", 
          \(\(-A\^2\)\ E\^\(2\ I\ T\_0\)\ \[Alpha]\_2 - 
            2\ A\ A\&_\ \[Alpha]\_2 - 
            E\^\(\(-2\)\ I\ T\_0\)\ A\&_\^2\ \[Alpha]\_2\)}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["whose particular solution can be expressed as", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(DSolve[order2Eqm, {u\_2[T\_0, T\_1, T\_2], v\_2[T\_0, T\_1, T\_2]}, 
      timeScales]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(DSolve::"pde" \( : \ \) 
      "Partial differential equation may not have a general solution. Try \
loading Calculus`DSolveIntegrals` to find special solutions."\)], "Message",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"DSolve", "[", 
      RowBox[{
        RowBox[{"{", 
          RowBox[{
            RowBox[{
              RowBox[{\(-v\_2[T\_0, T\_1, T\_2]\), "+", 
                RowBox[{
                  SubsuperscriptBox["u", "2", 
                    TagBox[\((1, 0, 0)\),
                      Derivative],
                    MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), 
                  "]"}]}], "==", "0"}], ",", 
            RowBox[{
              RowBox[{\(u\_2[T\_0, T\_1, T\_2]\), "+", 
                RowBox[{
                  SubsuperscriptBox["v", "2", 
                    TagBox[\((1, 0, 0)\),
                      Derivative],
                    MultilineFunction->None], "[", \(T\_0, T\_1, T\_2\), 
                  "]"}]}], "==", 
              \(\(-E\^\(2\ I\ T\_0\)\)\ A[T\_1, T\_2]\^2\ \[Alpha]\_2 - 
                2\ A[T\_1, T\_2]\ \[Alpha]\_2\ A\&_[T\_1, T\_2] - 
                E\^\(\(-2\)\ I\ T\_0\)\ \[Alpha]\_2\ 
                  A\&_[T\_1, T\_2]\^2\)}]}], "}"}], ",", 
        \({u\_2[T\_0, T\_1, T\_2], v\_2[T\_0, T\_1, T\_2]}\), ",", 
        \({T\_0, T\_1, T\_2}\)}], "]"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Here, directly applying ",
  StyleBox["DSolve",
    FontWeight->"Bold"],
  " to solve for ",
  StyleBox["order2Eqm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " does not work. Since the differential operators on the left-hand sides of \
",
  StyleBox["order2Eqm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " are essentially ordinary-differential operators in terms of ",
  Cell[BoxData[
      \(TraditionalForm\`T\_0\)]],
  ", we can first transform ",
  StyleBox["order2Eqm",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into an ordinary-differential-equation form and then use ",
  StyleBox["DSolve",
    FontWeight->"Bold"],
  " to obtain the particular solution as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol2pForm = 
      \(\(DSolve[
                order2Eqm /. {u\_2 -> \((u\_2[#1]\ &)\), 
                    v\_2 -> \((v\_2[#1]\ &)\)}, {u\_2[T\_0], v\_2[T\_0]}, 
                T\_0]\)[\([1]\)] /. C[_] -> 0 // TrigToExp\) // ExpandAll\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_2[T\_0] \[Rule] 
        1\/3\ E\^\(2\ I\ T\_0\)\ A[T\_1, T\_2]\^2\ \[Alpha]\_2 - 
          2\ A[T\_1, T\_2]\ \[Alpha]\_2\ A\&_[T\_1, T\_2] + 
          1\/3\ E\^\(\(-2\)\ I\ T\_0\)\ \[Alpha]\_2\ A\&_[T\_1, T\_2]\^2, 
      v\_2[T\_0] \[Rule] 
        2\/3\ I\ E\^\(2\ I\ T\_0\)\ A[T\_1, T\_2]\^2\ \[Alpha]\_2 - 
          2\/3\ I\ E\^\(\(-2\)\ I\ T\_0\)\ \[Alpha]\_2\ A\&_[T\_1, T\_2]\^2}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
We can then express the solution of the second-order equations in a pure \
function form as\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2Sol = {
        u\_2 -> Function[{T\_0, T\_1, T\_2}, 
            sol2pForm[\([1, 2]\)] // Evaluate], 
        v\_2 -> Function[{T\_0, T\_1, T\_2}, 
            sol2pForm[\([2, 2]\)] // Evaluate]}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_2 \[Rule] 
        Function[{T\_0, T\_1, T\_2}, 
          1\/3\ E\^\(2\ I\ T\_0\)\ A[T\_1, T\_2]\^2\ \[Alpha]\_2 - 
            2\ A[T\_1, T\_2]\ \[Alpha]\_2\ A\&_[T\_1, T\_2] + 
            1\/3\ E\^\(\(-2\)\ I\ T\_0\)\ \[Alpha]\_2\ A\&_[T\_1, T\_2]\^2], 
      v\_2 \[Rule] 
        Function[{T\_0, T\_1, T\_2}, 
          2\/3\ I\ E\^\(2\ I\ T\_0\)\ A[T\_1, T\_2]\^2\ \[Alpha]\_2 - 
            2\/3\ I\ E\^\(\(-2\)\ I\ T\_0\)\ \[Alpha]\_2\ 
              A\&_[T\_1, T\_2]\^2]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Third-Order Equations", "Subsubsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the first- and second-order solutions into the third-order \
equations, ",
  StyleBox["eqOrder[3]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order3Eq = \(eqOrder[3] /. order1Sol\) /. order2Sol // ExpandAll; \n
    order3Eq /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            InterpretationBox[\(D\_0\[InvisibleSpace]u\_3\),
              SequenceForm[ 
                Subscript[ D, 0], 
                Subscript[ u, 3]],
              Editable->False], "-", \(v\_3\)}], "==", 
          RowBox[{
            RowBox[{\(-E\^\(I\ T\_0\)\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_2\[InvisibleSpace]A\),
                  SequenceForm[ 
                    Subscript[ D, 2], A],
                  Editable->False], ")"}]}], "-", 
            RowBox[{\(E\^\(\(-I\)\ T\_0\)\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_2\[InvisibleSpace]A\&_\),
                  SequenceForm[ 
                    Subscript[ D, 2], 
                    OverBar[ A]],
                  Editable->False], ")"}]}], "-", 
            RowBox[{\(2\/3\), " ", "A", " ", \(E\^\(2\ I\ T\_0\)\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\),
                  SequenceForm[ 
                    Subscript[ D, 1], A],
                  Editable->False], ")"}], " ", \(\[Alpha]\_2\)}], "+", 
            RowBox[{"2", " ", \(A\&_\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\),
                  SequenceForm[ 
                    Subscript[ D, 1], A],
                  Editable->False], ")"}], " ", \(\[Alpha]\_2\)}], "+", 
            RowBox[{"2", " ", "A", " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    OverBar[ A]],
                  Editable->False], ")"}], " ", \(\[Alpha]\_2\)}], "-", 
            RowBox[{
            \(2\/3\), " ", \(E\^\(\(-2\)\ I\ T\_0\)\), " ", \(A\&_\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    OverBar[ A]],
                  Editable->False], ")"}], " ", \(\[Alpha]\_2\)}]}]}], ",", 
        RowBox[{
          RowBox[{
            InterpretationBox[\(D\_0\[InvisibleSpace]v\_3\),
              SequenceForm[ 
                Subscript[ D, 0], 
                Subscript[ v, 3]],
              Editable->False], "+", \(u\_3\)}], "==", 
          RowBox[{
            RowBox[{\(-I\), " ", \(E\^\(I\ T\_0\)\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_2\[InvisibleSpace]A\),
                  SequenceForm[ 
                    Subscript[ D, 2], A],
                  Editable->False], ")"}]}], "+", 
            RowBox[{"I", " ", \(E\^\(\(-I\)\ T\_0\)\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_2\[InvisibleSpace]A\&_\),
                  SequenceForm[ 
                    Subscript[ D, 2], 
                    OverBar[ A]],
                  Editable->False], ")"}]}], "-", 
            RowBox[{
            \(4\/3\), " ", "I", " ", "A", " ", \(E\^\(2\ I\ T\_0\)\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\),
                  SequenceForm[ 
                    Subscript[ D, 1], A],
                  Editable->False], ")"}], " ", \(\[Alpha]\_2\)}], "+", 
            RowBox[{
            \(4\/3\), " ", "I", " ", \(E\^\(\(-2\)\ I\ T\_0\)\), " ", 
              \(A\&_\), " ", 
              RowBox[{"(", 
                InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\),
                  SequenceForm[ 
                    Subscript[ D, 1], 
                    OverBar[ A]],
                  Editable->False], ")"}], " ", \(\[Alpha]\_2\)}], "-", 
            \(2\/3\ A\^3\ E\^\(3\ I\ T\_0\)\ \[Alpha]\_2\%2\), "+", 
            \(10\/3\ A\^2\ E\^\(I\ T\_0\)\ A\&_\ \[Alpha]\_2\%2\), "+", 
            \(10\/3\ A\ E\^\(\(-I\)\ T\_0\)\ A\&_\^2\ \[Alpha]\_2\%2\), "-", 
            \(2\/3\ E\^\(\(-3\)\ I\ T\_0\)\ A\&_\^3\ \[Alpha]\_2\%2\), "-", 
            \(A\^3\ E\^\(3\ I\ T\_0\)\ \[Alpha]\_3\), "-", 
            \(3\ A\^2\ E\^\(I\ T\_0\)\ A\&_\ \[Alpha]\_3\), "-", 
            \(3\ A\ E\^\(\(-I\)\ T\_0\)\ A\&_\^2\ \[Alpha]\_3\), "-", 
            \(E\^\(\(-3\)\ I\ T\_0\)\ A\&_\^3\ \[Alpha]\_3\)}]}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To eliminate the terms that lead to secular terms from ",
  StyleBox["order3Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we calculate the vector proportional to ",
  Cell[BoxData[
      \(TraditionalForm\`E\^\(I\ T\_0\)\)]],
  " in their right-hand sides and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(STerms2 = 
      \(Coefficient[#[\([2]\)], Exp[I\ T\_0]]\ &\)\ /@\ order3Eq\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{"-", 
          RowBox[{
            SuperscriptBox["A", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], ",", 
        RowBox[{
        \(10\/3\ A[T\_1, T\_2]\^2\ \[Alpha]\_2\%2\ A\&_[T\_1, T\_2]\), "-", 
          \(3\ A[T\_1, T\_2]\^2\ \[Alpha]\_3\ A\&_[T\_1, T\_2]\), "-", 
          RowBox[{"I", " ", 
            RowBox[{
              SuperscriptBox["A", 
                TagBox[\((0, 1)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}]}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["Hence, the solvability condition at this order is", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond2 = Conjugate[leftVec] . STerms2 == 0 // ExpandAll\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(\(-\(10\/3\)\)\ I\ A[T\_1, T\_2]\^2\ \[Alpha]\_2\%2\ 
          A\&_[T\_1, T\_2]\), "+", 
        \(3\ I\ A[T\_1, T\_2]\^2\ \[Alpha]\_3\ A\&_[T\_1, T\_2]\), "-", 
        RowBox[{"2", " ", 
          RowBox[{
            SuperscriptBox["A", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
This solvability condition is the same as that obtained by attacking the \
second-order form of the governing equation.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["3.4.3 First-Order Complex-Valued System", "Subsection",
  CellMargins->{{Inherited, 17.4375}, {Inherited, Inherited}},
  Evaluatable->False,
  TextAlignment->Left,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],

Cell[TextData[{
  "In this section, we first transform ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into a single first-order complex-valued equation using the \
transformation"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(transfRule = {u[t] -> \[Zeta][t] + \[Zeta]\&_[t], 
        \(u\^\[Prime]\)[t] -> I\ \((\[Zeta][t] - \[Zeta]\&_[t])\)}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["whose inverse is given by", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(zetaRule = 
      \(Solve[transfRule /. Rule -> Equal, {\[Zeta][t], \[Zeta]\&_[t]}]\)[
        \([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{\(\[Zeta][t]\), "\[Rule]", 
          RowBox[{\(1\/2\), " ", 
            RowBox[{"(", 
              RowBox[{\(u[t]\), "-", 
                RowBox[{"I", " ", 
                  RowBox[{
                    SuperscriptBox["u", "\[Prime]",
                      MultilineFunction->None], "[", "t", "]"}]}]}], 
              ")"}]}]}], ",", 
        RowBox[{\(\[Zeta]\&_[t]\), "\[Rule]", 
          RowBox[{\(1\/2\), " ", 
            RowBox[{"(", 
              RowBox[{\(u[t]\), "+", 
                RowBox[{"I", " ", 
                  RowBox[{
                    SuperscriptBox["u", "\[Prime]",
                      MultilineFunction->None], "[", "t", "]"}]}]}], 
              ")"}]}]}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`\[Zeta]\&\(\ _\)\)]],
  " is the complex conjugate of \[Zeta] as defined by the complex conjugate \
rule"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(conjugateRule = {A -> A\&_, A\&_ -> A, \[Zeta] -> \[Zeta]\&_, 
        \[Zeta]\&_ -> \[Zeta], Complex[0, n_] -> Complex[0, \(-n\)]}; \)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "It follows from ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that the acceleration is given by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(acceleration = \(Solve[eq31g, \(u\^\[DoublePrime]\)[t]]\)[\([1]\)]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          SuperscriptBox["u", "\[DoublePrime]",
            MultilineFunction->None], "[", "t", "]"}], "\[Rule]", 
        \(\(-u[t]\) - \[Alpha]\_2\ u[t]\^2 - \[Alpha]\_3\ u[t]\^3\)}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Differentiating ",
  StyleBox["zetaRule[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " once with respect to ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  " and using ",
  StyleBox["acceleration",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["transfRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the following first-order complex-valued equation:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq343a = 
      \(\(D[zetaRule[\([1]\)], t] /. acceleration\) /. transfRule\) /. 
          Rule -> Equal // ExpandAll\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["\[Zeta]", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(I\ \[Zeta][t] + 1\/2\ I\ \[Alpha]\_2\ \[Zeta][t]\^2 + 
        1\/2\ I\ \[Alpha]\_3\ \[Zeta][t]\^3 + 
        I\ \[Alpha]\_2\ \[Zeta][t]\ \[Zeta]\&_[t] + 
        3\/2\ I\ \[Alpha]\_3\ \[Zeta][t]\^2\ \[Zeta]\&_[t] + 
        1\/2\ I\ \[Alpha]\_2\ \[Zeta]\&_[t]\^2 + 
        3\/2\ I\ \[Alpha]\_3\ \[Zeta][t]\ \[Zeta]\&_[t]\^2 + 
        1\/2\ I\ \[Alpha]\_3\ \[Zeta]\&_[t]\^3\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine a second-order uniform expansion of the solution of ",
  StyleBox["eq343a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " using the method of multiple scales, we first introduce"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"multiScales", "=", 
        RowBox[{"{", 
          RowBox[{
          \(\[Zeta][t] -> \[Zeta][T\_0, T\_1, T\_2]\), ",", 
            \(\[Zeta]\&_[t] -> \[Zeta]\&_[T\_0, T\_1, T\_2]\), ",", 
            RowBox[{
              RowBox[{
                SuperscriptBox["\[Zeta]", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}], "->", 
              \(\(dt[1]\)[\[Zeta][T\_0, T\_1, T\_2]]\)}]}], "}"}]}], ";"}]], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "and then expand \[Zeta] and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Zeta]\&_\)]],
  " in the form"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(solRule = {
        \[Zeta] -> 
          \((Sum[\[Epsilon]\^j\ \[Zeta]\_j[#1, #2, #3], {j, 3}]\ &)\), 
        \[Zeta]\&_ -> 
          \((Sum[\(\[Epsilon]\^j\) \[Zeta]\&_\_j[#1, #2, #3], {j, 3}]\ &)\)}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting ",
  StyleBox["multiScales",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["solRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eq343a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result of small \[Epsilon], and discarding terms of order \
higher than ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Epsilon]\^3, \)\)]],
  " we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eq343b = 
      \((\(eq343a /. multiScales\) /. solRule // ExpandAll)\) /. 
        \[Epsilon]\^\(n_ /; n > 3\) -> 0; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[
"Equating coefficients of like powers of \[Epsilon] yields"], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eqEps = 
      \(CoefficientList[Subtract\ @@\ eq343b, \[Epsilon]] == 0 // Thread\) // 
        Rest; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
To place the linear operator on one side and the rest of the terms on the \
other, we define\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(eqOrder[i_] := 
      \(\((eqEps[\([1, 1]\)] /. \[Zeta]\_1 -> \[Zeta]\_i)\) - #\ &\)\ /@\ \ 
          eqEps[\([i]\)] // Reverse\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Using ",
  StyleBox["eqOrder[i]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["displayRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we rewrite ",
  StyleBox["eqEps",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a concise way as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(Array[eqOrder, 3] /. displayRule // TableForm\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    InterpretationBox[GridBox[{
          {
            RowBox[{
              RowBox[{
                InterpretationBox[\(D\_0\[InvisibleSpace]\[Zeta]\_1\),
                  SequenceForm[ 
                    Subscript[ D, 0], 
                    Subscript[ \[Zeta], 1]],
                  Editable->False], "-", \(I\ \[Zeta]\_1\)}], "==", "0"}]},
          {
            RowBox[{
              RowBox[{
                InterpretationBox[\(D\_0\[InvisibleSpace]\[Zeta]\_2\),
                  SequenceForm[ 
                    Subscript[ D, 0], 
                    Subscript[ \[Zeta], 2]],
                  Editable->False], "-", \(I\ \[Zeta]\_2\)}], "==", 
              RowBox[{
                RowBox[{"-", 
                  RowBox[{"(", 
                    InterpretationBox[\(D\_1\[InvisibleSpace]\[Zeta]\_1\),
                      SequenceForm[ 
                        Subscript[ D, 1], 
                        Subscript[ \[Zeta], 1]],
                      Editable->False], ")"}]}], "+", 
                \(1\/2\ I\ \[Alpha]\_2\ \[Zeta]\_1\%2\), "+", 
                \(I\ \[Alpha]\_2\ \[Zeta]\_1\ \[Zeta]\&_\_1\), "+", 
                \(1\/2\ I\ \[Alpha]\_2\ \[Zeta]\&_\_1\%2\)}]}]},
          {
            RowBox[{
              RowBox[{
                InterpretationBox[\(D\_0\[InvisibleSpace]\[Zeta]\_3\),
                  SequenceForm[ 
                    Subscript[ D, 0], 
                    Subscript[ \[Zeta], 3]],
                  Editable->False], "-", \(I\ \[Zeta]\_3\)}], "==", 
              RowBox[{
                RowBox[{"-", 
                  RowBox[{"(", 
                    InterpretationBox[\(D\_1\[InvisibleSpace]\[Zeta]\_2\),
                      SequenceForm[ 
                        Subscript[ D, 1], 
                        Subscript[ \[Zeta], 2]],
                      Editable->False], ")"}]}], "-", 
                InterpretationBox[\(D\_2\[InvisibleSpace]\[Zeta]\_1\),
                  SequenceForm[ 
                    Subscript[ D, 2], 
                    Subscript[ \[Zeta], 1]],
                  Editable->False], "+", 
                \(1\/2\ I\ \[Alpha]\_3\ \[Zeta]\_1\%3\), "+", 
                \(I\ \[Alpha]\_2\ \[Zeta]\_1\ \[Zeta]\_2\), "+", 
                \(3\/2\ I\ \[Alpha]\_3\ \[Zeta]\_1\%2\ \[Zeta]\&_\_1\), "+", 
                \(I\ \[Alpha]\_2\ \[Zeta]\_2\ \[Zeta]\&_\_1\), "+", 
                \(3\/2\ I\ \[Alpha]\_3\ \[Zeta]\_1\ \[Zeta]\&_\_1\%2\), "+", 
                \(1\/2\ I\ \[Alpha]\_3\ \[Zeta]\&_\_1\%3\), "+", 
                \(I\ \[Alpha]\_2\ \[Zeta]\_1\ \[Zeta]\&_\_2\), "+", 
                \(I\ \[Alpha]\_2\ \[Zeta]\&_\_1\ \[Zeta]\&_\_2\)}]}]}
          },
        RowSpacings->1,
        ColumnSpacings->3,
        RowAlignments->Baseline,
        ColumnAlignments->{Left}],
      TableForm[ {
        Equal[ 
          Plus[ 
            SequenceForm[ 
              Subscript[ D, 0], 
              Subscript[ \[Zeta], 1]], 
            Times[ 
              Complex[ 0, -1], 
              Subscript[ \[Zeta], 1]]], 0], 
        Equal[ 
          Plus[ 
            SequenceForm[ 
              Subscript[ D, 0], 
              Subscript[ \[Zeta], 2]], 
            Times[ 
              Complex[ 0, -1], 
              Subscript[ \[Zeta], 2]]], 
          Plus[ 
            Times[ -1, 
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ \[Zeta], 1]]], 
            Times[ 
              Complex[ 0, 
                Rational[ 1, 2]], 
              Subscript[ \[Alpha], 2], 
              Power[ 
                Subscript[ \[Zeta], 1], 2]], 
            Times[ 
              Complex[ 0, 1], 
              Subscript[ \[Alpha], 2], 
              Subscript[ \[Zeta], 1], 
              Subscript[ 
                OverBar[ \[Zeta]], 1]], 
            Times[ 
              Complex[ 0, 
                Rational[ 1, 2]], 
              Subscript[ \[Alpha], 2], 
              Power[ 
                Subscript[ 
                  OverBar[ \[Zeta]], 1], 2]]]], 
        Equal[ 
          Plus[ 
            SequenceForm[ 
              Subscript[ D, 0], 
              Subscript[ \[Zeta], 3]], 
            Times[ 
              Complex[ 0, -1], 
              Subscript[ \[Zeta], 3]]], 
          Plus[ 
            Times[ -1, 
              SequenceForm[ 
                Subscript[ D, 1], 
                Subscript[ \[Zeta], 2]]], 
            Times[ -1, 
              SequenceForm[ 
                Subscript[ D, 2], 
                Subscript[ \[Zeta], 1]]], 
            Times[ 
              Complex[ 0, 
                Rational[ 1, 2]], 
              Subscript[ \[Alpha], 3], 
              Power[ 
                Subscript[ \[Zeta], 1], 3]], 
            Times[ 
              Complex[ 0, 1], 
              Subscript[ \[Alpha], 2], 
              Subscript[ \[Zeta], 1], 
              Subscript[ \[Zeta], 2]], 
            Times[ 
              Complex[ 0, 
                Rational[ 3, 2]], 
              Subscript[ \[Alpha], 3], 
              Power[ 
                Subscript[ \[Zeta], 1], 2], 
              Subscript[ 
                OverBar[ \[Zeta]], 1]], 
            Times[ 
              Complex[ 0, 1], 
              Subscript[ \[Alpha], 2], 
              Subscript[ \[Zeta], 2], 
              Subscript[ 
                OverBar[ \[Zeta]], 1]], 
            Times[ 
              Complex[ 0, 
                Rational[ 3, 2]], 
              Subscript[ \[Alpha], 3], 
              Subscript[ \[Zeta], 1], 
              Power[ 
                Subscript[ 
                  OverBar[ \[Zeta]], 1], 2]], 
            Times[ 
              Complex[ 0, 
                Rational[ 1, 2]], 
              Subscript[ \[Alpha], 3], 
              Power[ 
                Subscript[ 
                  OverBar[ \[Zeta]], 1], 3]], 
            Times[ 
              Complex[ 0, 1], 
              Subscript[ \[Alpha], 2], 
              Subscript[ \[Zeta], 1], 
              Subscript[ 
                OverBar[ \[Zeta]], 2]], 
            Times[ 
              Complex[ 0, 1], 
              Subscript[ \[Alpha], 2], 
              Subscript[ 
                OverBar[ \[Zeta]], 1], 
              Subscript[ 
                OverBar[ \[Zeta]], 2]]]]}]]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The general solution of the first-order equation, ",
  StyleBox["eqOrder[1]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", can be expressed in a pure function form as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol1 = 
      \(DSolve[eqOrder[1], \[Zeta]\_1, timeScales]\)[\([1]\)] /. 
        C[1] -> A\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Zeta]\_1 \[Rule] \((E\^\(I\ #1\)\ A[#2, #3]&)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["whose complex conjugate is", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ccsol1 = sol1 /. conjugateRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Zeta]\&_\_1 \[Rule] 
        \((E\^\(Complex[0, \(-1\)]\ #1\)\ A\&_[#2, #3]&)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting the first-order solution into the second-order equation, ",
  StyleBox["eqOrder[2]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2Eq = \(eqOrder[2] /. sol1\) /. ccsol1; \n
    order2Eq /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        InterpretationBox[\(D\_0\[InvisibleSpace]\[Zeta]\_2\),
          SequenceForm[ 
            Subscript[ D, 0], 
            Subscript[ \[Zeta], 2]],
          Editable->False], "-", \(I\ \[Zeta]\_2\)}], "==", 
      RowBox[{
        RowBox[{\(-E\^\(I\ T\_0\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\),
              SequenceForm[ 
                Subscript[ D, 1], A],
              Editable->False], ")"}]}], "+", 
        \(1\/2\ I\ A\^2\ E\^\(2\ I\ T\_0\)\ \[Alpha]\_2\), "+", 
        \(I\ A\ A\&_\ \[Alpha]\_2\), "+", 
        \(1\/2\ I\ E\^\(\(-2\)\ I\ T\_0\)\ A\&_\^2\ \[Alpha]\_2\)}]}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Eliminating the terms that lead to seculr terms in ",
  Cell[BoxData[
      \(TraditionalForm\`\[Zeta]\_2\)]],
  " from ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond1 = Coefficient[order2Eq[\([2]\)], E\^\(I\ T\_0\)] == 0\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"-", 
        RowBox[{
          SuperscriptBox["A", 
            TagBox[\((1, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["or", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    RowBox[{"SCond1Rule", "=", 
      RowBox[{
        RowBox[{"Solve", "[", 
          RowBox[{"SCond1", ",", 
            RowBox[{
              SuperscriptBox["A", 
                TagBox[\((1, 0)\),
                  Derivative],
                MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}]}], 
          "]"}], "[", \([1]\), "]"}]}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          SuperscriptBox["A", 
            TagBox[\((1, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(T\_1, T\_2\), "]"}], "\[Rule]", 
        "0"}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "With this solvability condition, the particular solution of ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be expressed in a pure function form as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(sol2 = 
      \(DSolve[order2Eq /. SCond1Rule, \[Zeta]\_2, timeScales]\)[\([1]\)] /. 
        C[_] -> \((0\ &)\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Zeta]\_2 \[Rule] 
        \((E\^\(I\ #1\)\ \((0&)\)[#2, #3] + 
              1\/6\ E\^\(\(-2\)\ I\ #1\)\ \[Alpha]\_2\ 
                \((3\ E\^\(4\ I\ #1\)\ A[#2, #3]\^2 - 
                    6\ E\^\(2\ I\ #1\)\ A[#2, #3]\ A\&_[#2, #3] - 
                    A\&_[#2, #3]\^2)\)&)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["whose complex conjugate is", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(ccsol2 = sol2 /. conjugateRule; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the first- and second-order solutions in the third-order \
equation, ",
  StyleBox["eqOrder[3]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order3Eq = 
      \(\(\(eqOrder[3] /. sol1\) /. ccsol1\) /. sol2\) /. ccsol2 // 
        ExpandAll; \norder3Eq /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        InterpretationBox[\(D\_0\[InvisibleSpace]\[Zeta]\_3\),
          SequenceForm[ 
            Subscript[ D, 0], 
            Subscript[ \[Zeta], 3]],
          Editable->False], "-", \(I\ \[Zeta]\_3\)}], "==", 
      RowBox[{
        RowBox[{\(-E\^\(I\ T\_0\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_2\[InvisibleSpace]A\),
              SequenceForm[ 
                Subscript[ D, 2], A],
              Editable->False], ")"}]}], "-", 
        RowBox[{"A", " ", \(E\^\(2\ I\ T\_0\)\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\),
              SequenceForm[ 
                Subscript[ D, 1], A],
              Editable->False], ")"}], " ", \(\[Alpha]\_2\)}], "+", 
        RowBox[{\(A\&_\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\),
              SequenceForm[ 
                Subscript[ D, 1], A],
              Editable->False], ")"}], " ", \(\[Alpha]\_2\)}], "+", 
        RowBox[{"A", " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\),
              SequenceForm[ 
                Subscript[ D, 1], 
                OverBar[ A]],
              Editable->False], ")"}], " ", \(\[Alpha]\_2\)}], "+", 
        RowBox[{
        \(1\/3\), " ", \(E\^\(\(-2\)\ I\ T\_0\)\), " ", \(A\&_\), " ", 
          RowBox[{"(", 
            InterpretationBox[\(D\_1\[InvisibleSpace]A\&_\),
              SequenceForm[ 
                Subscript[ D, 1], 
                OverBar[ A]],
              Editable->False], ")"}], " ", \(\[Alpha]\_2\)}], "+", 
        \(1\/3\ I\ A\^3\ E\^\(3\ I\ T\_0\)\ \[Alpha]\_2\%2\), "-", 
        \(5\/3\ I\ A\^2\ E\^\(I\ T\_0\)\ A\&_\ \[Alpha]\_2\%2\), "-", 
        \(5\/3\ I\ A\ E\^\(\(-I\)\ T\_0\)\ A\&_\^2\ \[Alpha]\_2\%2\), "+", 
        \(1\/3\ I\ E\^\(\(-3\)\ I\ T\_0\)\ A\&_\^3\ \[Alpha]\_2\%2\), "+", 
        \(1\/2\ I\ A\^3\ E\^\(3\ I\ T\_0\)\ \[Alpha]\_3\), "+", 
        \(3\/2\ I\ A\^2\ E\^\(I\ T\_0\)\ A\&_\ \[Alpha]\_3\), "+", 
        \(3\/2\ I\ A\ E\^\(\(-I\)\ T\_0\)\ A\&_\^2\ \[Alpha]\_3\), "+", 
        \(1\/2\ I\ E\^\(\(-3\)\ I\ T\_0\)\ A\&_\^3\ \[Alpha]\_3\)}]}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Eliminating the terms that produce secular terms in ",
  Cell[BoxData[
      \(TraditionalForm\`\[Zeta]\_3\)]],
  " from ",
  StyleBox["order3Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " demands the following condition"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(SCond2 = Coefficient[order3Eq[\([2]\)], E\^\(I\ T\_0\)] == 0; \n
    SCond2 /. displayRule\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{"-", 
          RowBox[{"(", 
            InterpretationBox[\(D\_2\[InvisibleSpace]A\),
              SequenceForm[ 
                Subscript[ D, 2], A],
              Editable->False], ")"}]}], "-", 
        \(5\/3\ I\ A\^2\ A\&_\ \[Alpha]\_2\%2\), "+", 
        \(3\/2\ I\ A\^2\ A\&_\ \[Alpha]\_3\)}], "==", "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "which is in full agreement with that obtained by attacking directly ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["3.5 The Method of Averaging", "Section",
  CellMargins->{{Inherited, 17.4375}, {Inherited, Inherited}},
  Evaluatable->False,
  TextAlignment->Left,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],

Cell[TextData[{
  "As before, we use the method of variation of parameters to transform the \
dependent variable ",
  Cell[BoxData[
      \(TraditionalForm\`u\)]],
  StyleBox[" ",
    FontSlant->"Italic"],
  "into ",
  StyleBox["a",
    FontSlant->"Italic"],
  " and \[Beta] according to"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eq35a = u[t] == \[Epsilon]\ a\ Cos[t + \[Beta]]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["subject to the constraint", "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq35b = D[eq35a, t]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["u", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(\(-a\)\ \[Epsilon]\ Sin[t + \[Beta]]\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  StyleBox["a",
    FontSlant->"Italic"],
  " and \[Beta] are functions of ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]]
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(tdependentRule = {a -> a[t], \[Beta] -> \[Beta][t]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "and \[Epsilon] is a small dimensionless parameter that is a measure of the \
amplitude of oscillation. Differentiating ",
  StyleBox["eq35a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " with respect to ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq35c = D[eq35a /. tdependentRule, t]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["u", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      RowBox[{
        RowBox[{"\[Epsilon]", " ", \(Cos[t + \[Beta][t]]\), " ", 
          RowBox[{
            SuperscriptBox["a", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}]}], "-", 
        RowBox[{
        "\[Epsilon]", " ", \(a[t]\), " ", \(Sin[t + \[Beta][t]]\), " ", 
          RowBox[{"(", 
            RowBox[{"1", "+", 
              RowBox[{
                SuperscriptBox["\[Beta]", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}], ")"}]}]}]}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Comparing ",
  StyleBox["eq35b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["eq35c",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we conclude that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(cond35a = 
      Expand[eq35c[\([2]\)] - \((eq35b[\([2]\)] /. tdependentRule)\)] == 0
        \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        RowBox[{"\[Epsilon]", " ", \(Cos[t + \[Beta][t]]\), " ", 
          RowBox[{
            SuperscriptBox["a", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}]}], "-", 
        RowBox[{
        "\[Epsilon]", " ", \(a[t]\), " ", \(Sin[t + \[Beta][t]]\), " ", 
          RowBox[{
            SuperscriptBox["\[Beta]", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}]}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Differentiating ",
  StyleBox["eq35b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " once with respect to ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  " yields"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq35d = D[eq35b /. tdependentRule, t]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["u", "\[DoublePrime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      RowBox[{
        RowBox[{\(-\[Epsilon]\), " ", \(Sin[t + \[Beta][t]]\), " ", 
          RowBox[{
            SuperscriptBox["a", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}]}], "-", 
        RowBox[{
        "\[Epsilon]", " ", \(a[t]\), " ", \(Cos[t + \[Beta][t]]\), " ", 
          RowBox[{"(", 
            RowBox[{"1", "+", 
              RowBox[{
                SuperscriptBox["\[Beta]", "\[Prime]",
                  MultilineFunction->None], "[", "t", "]"}]}], ")"}]}]}]}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting ",
  StyleBox["eq35a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["eq35d",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(cond35b = 
      eq31g /. \(({eq35a /. tdependentRule, eq35d} /. Equal -> Rule)\) // 
        ExpandAll\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(\[Epsilon]\^2\ a[t]\^2\ Cos[t + \[Beta][t]]\^2\ \[Alpha]\_2\), "+", 
        \(\[Epsilon]\^3\ a[t]\^3\ Cos[t + \[Beta][t]]\^3\ \[Alpha]\_3\), 
        "-", 
        RowBox[{"\[Epsilon]", " ", \(Sin[t + \[Beta][t]]\), " ", 
          RowBox[{
            SuperscriptBox["a", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}]}], "-", 
        RowBox[{
        "\[Epsilon]", " ", \(a[t]\), " ", \(Cos[t + \[Beta][t]]\), " ", 
          RowBox[{
            SuperscriptBox["\[Beta]", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}]}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Solving ",
  StyleBox["cond35a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["cond35b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " for ",
  Cell[BoxData[
      \(TraditionalForm\`\(a\^\[Prime]\)[t]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Beta]\^\[Prime]\)[t]\)]],
  ", we have"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(transformedEq = 
      \(Solve[{cond35a, cond35b}, {\(a\^\[Prime]\)[t], 
                \(\[Beta]\^\[Prime]\)[t]}]\)[\([1]\)] /. Rule -> Equal // 
        Simplify\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SuperscriptBox["a", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], "==", 
          \(\[Epsilon]\ a[t]\^2\ Cos[t + \[Beta][t]]\^2\ Sin[t + \[Beta][t]]\ 
            \((\[Alpha]\_2 + 
                \[Epsilon]\ a[t]\ Cos[t + \[Beta][t]]\ \[Alpha]\_3)\)\)}], 
        ",", 
        RowBox[{
          RowBox[{
            SuperscriptBox["\[Beta]", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], "==", 
          \(\[Epsilon]\ a[t]\ Cos[t + \[Beta][t]]\^3\ 
            \((\[Alpha]\_2 + 
                \[Epsilon]\ a[t]\ Cos[t + \[Beta][t]]\ \[Alpha]\_3)\)\)}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`a\)]],
  " is assumed to be different from zero in arriving at ",
  StyleBox["transformedEq[[2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". Using trigonometric identities, we expand the right-hand sides of ",
  StyleBox["transformedEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in Fourier series and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ampEq1 = TrigReduce\ /@\ transformedEq[\([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["a", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(1\/8\ \((
          2\ \[Epsilon]\ a[t]\^2\ Sin[t + \[Beta][t]]\ \[Alpha]\_2 + 
            2\ \[Epsilon]\ a[t]\^2\ Sin[3\ t + 3\ \[Beta][t]]\ \[Alpha]\_2 + 
            2\ \[Epsilon]\^2\ a[t]\^3\ Sin[2\ t + 2\ \[Beta][t]]\ 
              \[Alpha]\_3 + 
            \[Epsilon]\^2\ a[t]\^3\ Sin[4\ t + 4\ \[Beta][t]]\ \[Alpha]\_3)
          \)\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phaseEq1 = TrigReduce\ /@\ transformedEq[\([2]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["\[Beta]", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(1\/8\ \((
          6\ \[Epsilon]\ a[t]\ Cos[t + \[Beta][t]]\ \[Alpha]\_2 + 
            2\ \[Epsilon]\ a[t]\ Cos[3\ t + 3\ \[Beta][t]]\ \[Alpha]\_2 + 
            3\ \[Epsilon]\^2\ a[t]\^2\ \[Alpha]\_3 + 
            4\ \[Epsilon]\^2\ a[t]\^2\ Cos[2\ t + 2\ \[Beta][t]]\ 
              \[Alpha]\_3 + 
            \[Epsilon]\^2\ a[t]\^2\ Cos[4\ t + 4\ \[Beta][t]]\ \[Alpha]\_3)
          \)\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Since ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\)]],
  " is small, ",
  Cell[BoxData[
      \(TraditionalForm\`\(a\^\[Prime]\)[t]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Beta]\^\[Prime]\)[t]\)]],
  " are slowly varying functions of ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  ". Then, according to the method of averaging, we can neglect the \
fast-varying terms in ",
  StyleBox["ampEq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phaseEq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and obtain the following averaged equations:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ampAvgEq = ampEq1 /. Sin[_. \ t + _] -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["a", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phaseAvgEq = phaseEq1 /. Cos[_. \ t + _] -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["\[Beta]", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(3\/8\ \[Epsilon]\^2\ a[t]\^2\ \[Alpha]\_3\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Whereas ",
  StyleBox["ampAvgEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " agrees with ",
  StyleBox["ampEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["phaseAvgEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " does not agree with ",
  StyleBox["phaseEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " obtained in the preceding section by using the method of multiple scales. \
There is a term ",
  Cell[BoxData[
      \(TraditionalForm\`\(5\/12\) \[Epsilon]\^2\ a\^2\ \[Alpha]\_2\%2\)]],
  " missing from ",
  StyleBox["phaseAvgEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". Following the details of the solution in the preceding section, one \
finds that this term is the result of the interaction of the first- and \
second-order approximations. This interaction was not taken into account in \
arriving at ",
  StyleBox["ampAvgEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phaseAvgEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". To include the effect of this interaction, we need to carry out the \
solutions of ",
  StyleBox["ampEq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phaseEq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " to higher order. This is accomplished by using the generalized method of \
averaging, which is discussed next, or its variant the \
Krylov-Bogoliubov-Mitropolsky technique, which is discussed in Section 3.7."
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["3.6 The Generalized Method of Averaging", "Section",
  CellMargins->{{Inherited, 17.4375}, {Inherited, Inherited}},
  Evaluatable->False,
  TextAlignment->Left,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],

Cell[TextData[{
  "To apply this method, we introduce the variable ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi] = t + \[Beta]\)]],
  " and rewrite ",
  StyleBox["ampEq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phaseEq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " as"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ampEq2 = 
      ampEq1 /. 
        Sin[n_. \ t + a_] :> 
          Sin[Expand[n\ \((\[Phi][t] - \[Beta][t])\) + a]]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["a", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(1\/8\ \((
          2\ \[Epsilon]\ a[t]\^2\ Sin[\[Phi][t]]\ \[Alpha]\_2 + 
            2\ \[Epsilon]\ a[t]\^2\ Sin[3\ \[Phi][t]]\ \[Alpha]\_2 + 
            2\ \[Epsilon]\^2\ a[t]\^3\ Sin[2\ \[Phi][t]]\ \[Alpha]\_3 + 
            \[Epsilon]\^2\ a[t]\^3\ Sin[4\ \[Phi][t]]\ \[Alpha]\_3)\)\)}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phaseEq2 = 
      \(\[Phi]\^\[Prime]\)[t] == 1 + phaseEq1[\([2]\)] /. 
        Cos[n_. \ t + a_] :> 
          Cos[Expand[n\ \((\[Phi][t] - \[Beta][t])\) + a]]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["\[Phi]", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(1 + 1\/8\ 
          \((6\ \[Epsilon]\ a[t]\ Cos[\[Phi][t]]\ \[Alpha]\_2 + 
              2\ \[Epsilon]\ a[t]\ Cos[3\ \[Phi][t]]\ \[Alpha]\_2 + 
              3\ \[Epsilon]\^2\ a[t]\^2\ \[Alpha]\_3 + 
              4\ \[Epsilon]\^2\ a[t]\^2\ Cos[2\ \[Phi][t]]\ \[Alpha]\_3 + 
              \[Epsilon]\^2\ a[t]\^2\ Cos[4\ \[Phi][t]]\ \[Alpha]\_3)\)\)}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "We seek approximate solutions to ",
  StyleBox["ampEq2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phaseEq2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in the form"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(asol = 
      a[t] == a\_0[t] + 
          Sum[\[Epsilon]\^i\ a\_i[a\_0[t], \[Phi]\_0[t]], {i, 2}]\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(a[t] == 
      a\_0[t] + \[Epsilon]\ a\_1[a\_0[t], \[Phi]\_0[t]] + 
        \[Epsilon]\^2\ a\_2[a\_0[t], \[Phi]\_0[t]]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phisol = 
      \[Phi][t] == 
        \[Phi]\_0[t] + 
          Sum[\[Epsilon]\^i\ \[Phi]\_i[a\_0[t], \[Phi]\_0[t]], {i, 2}]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\[Phi][t] == 
      \[Phi]\_0[t] + \[Epsilon]\ \[Phi]\_1[a\_0[t], \[Phi]\_0[t]] + 
        \[Epsilon]\^2\ \[Phi]\_2[a\_0[t], \[Phi]\_0[t]]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`a\_0[t]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_0[t]\)]],
  " are expanded in power series in \[Epsilon] as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(a0Eq = 
      \(a\_0\^\[Prime]\)[t] == Sum[\[Epsilon]\^i\ A\_i[a\_0[t]], {i, 2}]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SubsuperscriptBox["a", "0", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(\[Epsilon]\ A\_1[a\_0[t]] + \[Epsilon]\^2\ A\_2[a\_0[t]]\)}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phi0Eq = 
      \(\[Phi]\_0\^\[Prime]\)[t] == 
        1 + Sum[\[Epsilon]\^i\ \[CapitalPhi]\_i[a\_0[t]], {i, 2}]\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SubsuperscriptBox["\[Phi]", "0", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(1 + \[Epsilon]\ \[CapitalPhi]\_1[a\_0[t]] + 
        \[Epsilon]\^2\ \[CapitalPhi]\_2[a\_0[t]]\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The functions ",
  Cell[BoxData[
      \(TraditionalForm\`a\_1, a\_2, \[Ellipsis], a\_n\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_1, \[Phi]\_2, \[Ellipsis]\)]],
  ", ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_n\)]],
  " are fast varying functions of ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_0\)]],
  ", while it follows from ",
  StyleBox["a0Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phi0Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that ",
  Cell[BoxData[
      \(TraditionalForm\`a\_0\)]],
  ", and hence the ",
  Cell[BoxData[
      \(TraditionalForm\`A\_n\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPhi]\_n\)]],
  " are slowly varying functions of ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  "."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "To the second approximation, we differentiate ",
  Cell[BoxData[
      \(TraditionalForm\`a[t]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi][t]\)]],
  " once with respect to ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  ", use ",
  StyleBox["a0Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phi0Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expand the result for small \[Epsilon], discard terms of order higher \
than ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^2\)]],
  ", and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(aEq1 = 
      \((D[asol, t] /. \(({a0Eq, phi0Eq} /. Equal -> Rule)\) // ExpandAll)
          \) /. \[Epsilon]\^\(n_ /; n > 2\) -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["a", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      RowBox[{
      \(\[Epsilon]\ A\_1[a\_0[t]]\), "+", \(\[Epsilon]\^2\ A\_2[a\_0[t]]\), 
        "+", 
        RowBox[{"\[Epsilon]", " ", 
          RowBox[{
            SubsuperscriptBox["a", "1", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
            "]"}]}], "+", 
        RowBox[{\(\[Epsilon]\^2\), " ", \(\[CapitalPhi]\_1[a\_0[t]]\), " ", 
          RowBox[{
            SubsuperscriptBox["a", "1", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
            "]"}]}], "+", 
        RowBox[{\(\[Epsilon]\^2\), " ", 
          RowBox[{
            SubsuperscriptBox["a", "2", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
            "]"}]}], "+", 
        RowBox[{\(\[Epsilon]\^2\), " ", \(A\_1[a\_0[t]]\), " ", 
          RowBox[{
            SubsuperscriptBox["a", "1", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
            "]"}]}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phiEq1 = 
      \((D[phisol, t] /. \(({a0Eq, phi0Eq} /. Equal -> Rule)\) // ExpandAll)
          \) /. \[Epsilon]\^\(n_ /; n > 2\) -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["\[Phi]", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      RowBox[{
      "1", "+", \(\[Epsilon]\ \[CapitalPhi]\_1[a\_0[t]]\), "+", 
        \(\[Epsilon]\^2\ \[CapitalPhi]\_2[a\_0[t]]\), "+", 
        RowBox[{"\[Epsilon]", " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
            "]"}]}], "+", 
        RowBox[{\(\[Epsilon]\^2\), " ", \(\[CapitalPhi]\_1[a\_0[t]]\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
            "]"}]}], "+", 
        RowBox[{\(\[Epsilon]\^2\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "2", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
            "]"}]}], "+", 
        RowBox[{\(\[Epsilon]\^2\), " ", \(A\_1[a\_0[t]]\), " ", 
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", 
              TagBox[\((1, 0)\),
                Derivative],
              MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
            "]"}]}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Next, we substitute ",
  StyleBox["asol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phisol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["ampEq2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phaseEq2",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expand the right-hand sides for small \[Epsilon] keeping terms up to ",
  Cell[BoxData[
      \(TraditionalForm\`O(\[Epsilon]\^2)\)]],
  ", and rewrite their right-hand sides as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ampEqrhs = 
      Series[ampEq2[\([2]\)] /. \(({asol, phisol} /. Equal -> Rule)\), {
            \[Epsilon], 0, 2}] // Normal\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(1\/8\ \[Epsilon]\ 
        \((2\ Sin[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2 + 
            2\ Sin[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2)\) + 
      1\/8\ \[Epsilon]\^2\ 
        \((2\ Sin[2\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^3 + 
            Sin[4\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^3 + 
            2\ \[Alpha]\_2\ 
              \((2\ Sin[\[Phi]\_0[t]]\ a\_0[t]\ a\_1[a\_0[t], \[Phi]\_0[t]] + 
                  Cos[\[Phi]\_0[t]]\ a\_0[t]\^2\ 
                    \[Phi]\_1[a\_0[t], \[Phi]\_0[t]])\) + 
            2\ \[Alpha]\_2\ 
              \((2\ Sin[3\ \[Phi]\_0[t]]\ a\_0[t]\ 
                    a\_1[a\_0[t], \[Phi]\_0[t]] + 
                  3\ Cos[3\ \[Phi]\_0[t]]\ a\_0[t]\^2\ 
                    \[Phi]\_1[a\_0[t], \[Phi]\_0[t]])\))\)\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phaseEqrhs = 
      Series[phaseEq2[\([2]\)] /. \(({asol, phisol} /. Equal -> Rule)\), {
            \[Epsilon], 0, 2}] // Normal\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(1 + 
      1\/8\ \[Epsilon]\ 
        \((6\ Cos[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t] + 
            2\ Cos[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t])\) + 
      1\/8\ \[Epsilon]\^2\ 
        \((3\ \[Alpha]\_3\ a\_0[t]\^2 + 
            4\ Cos[2\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^2 + 
            Cos[4\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^2 + 
            6\ \[Alpha]\_2\ 
              \((Cos[\[Phi]\_0[t]]\ a\_1[a\_0[t], \[Phi]\_0[t]] - 
                  Sin[\[Phi]\_0[t]]\ a\_0[t]\ 
                    \[Phi]\_1[a\_0[t], \[Phi]\_0[t]])\) + 
            2\ \[Alpha]\_2\ 
              \((Cos[3\ \[Phi]\_0[t]]\ a\_1[a\_0[t], \[Phi]\_0[t]] - 
                  3\ Sin[3\ \[Phi]\_0[t]]\ a\_0[t]\ 
                    \[Phi]\_1[a\_0[t], \[Phi]\_0[t]])\))\)\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Equating coefficients of like powers of \[Epsilon] from ",
  StyleBox["aEq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["ampEqrhs",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["phiEq1",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and ",
  StyleBox["phaseEqrhs",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ampEq3 = 
      \(\(CoefficientList[#, \[Epsilon]]\ &\)\ /@\ 
            \((aEq1[\([2]\)] == ampEqrhs)\) // Thread\) // Rest\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(A\_1[a\_0[t]]\), "+", 
            RowBox[{
              SubsuperscriptBox["a", "1", 
                TagBox[\((0, 1)\),
                  Derivative],
                MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
              "]"}]}], "==", 
          \(1\/8\ \((
              2\ Sin[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2 + 
                2\ Sin[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2)\)\)}], 
        ",", 
        RowBox[{
          RowBox[{\(A\_2[a\_0[t]]\), "+", 
            RowBox[{\(\[CapitalPhi]\_1[a\_0[t]]\), " ", 
              RowBox[{
                SubsuperscriptBox["a", "1", 
                  TagBox[\((0, 1)\),
                    Derivative],
                  MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
                "]"}]}], "+", 
            RowBox[{
              SubsuperscriptBox["a", "2", 
                TagBox[\((0, 1)\),
                  Derivative],
                MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
              "]"}], "+", 
            RowBox[{\(A\_1[a\_0[t]]\), " ", 
              RowBox[{
                SubsuperscriptBox["a", "1", 
                  TagBox[\((1, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
                "]"}]}]}], "==", 
          \(1\/8\ \((
              2\ Sin[2\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^3 + 
                Sin[4\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^3 + 
                2\ \[Alpha]\_2\ 
                  \((2\ Sin[\[Phi]\_0[t]]\ a\_0[t]\ 
                        a\_1[a\_0[t], \[Phi]\_0[t]] + 
                      Cos[\[Phi]\_0[t]]\ a\_0[t]\^2\ 
                        \[Phi]\_1[a\_0[t], \[Phi]\_0[t]])\) + 
                2\ \[Alpha]\_2\ 
                  \((2\ Sin[3\ \[Phi]\_0[t]]\ a\_0[t]\ 
                        a\_1[a\_0[t], \[Phi]\_0[t]] + 
                      3\ Cos[3\ \[Phi]\_0[t]]\ a\_0[t]\^2\ 
                        \[Phi]\_1[a\_0[t], \[Phi]\_0[t]])\))\)\)}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phaseEq3 = 
      \(\(CoefficientList[#, \[Epsilon]]&\)\ /@\ 
            \((phiEq1[\([2]\)] == phaseEqrhs)\) // Thread\) // Rest\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(\[CapitalPhi]\_1[a\_0[t]]\), "+", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "1", 
                TagBox[\((0, 1)\),
                  Derivative],
                MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
              "]"}]}], "==", 
          \(1\/8\ \((
              6\ Cos[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t] + 
                2\ Cos[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t])\)\)}], ",", 
        RowBox[{
          RowBox[{\(\[CapitalPhi]\_2[a\_0[t]]\), "+", 
            RowBox[{\(\[CapitalPhi]\_1[a\_0[t]]\), " ", 
              RowBox[{
                SubsuperscriptBox["\[Phi]", "1", 
                  TagBox[\((0, 1)\),
                    Derivative],
                  MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
                "]"}]}], "+", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", 
                TagBox[\((0, 1)\),
                  Derivative],
                MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
              "]"}], "+", 
            RowBox[{\(A\_1[a\_0[t]]\), " ", 
              RowBox[{
                SubsuperscriptBox["\[Phi]", "1", 
                  TagBox[\((1, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
                "]"}]}]}], "==", 
          \(1\/8\ \((
              3\ \[Alpha]\_3\ a\_0[t]\^2 + 
                4\ Cos[2\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^2 + 
                Cos[4\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^2 + 
                6\ \[Alpha]\_2\ 
                  \((Cos[\[Phi]\_0[t]]\ a\_1[a\_0[t], \[Phi]\_0[t]] - 
                      Sin[\[Phi]\_0[t]]\ a\_0[t]\ 
                        \[Phi]\_1[a\_0[t], \[Phi]\_0[t]])\) + 
                2\ \[Alpha]\_2\ 
                  \((Cos[3\ \[Phi]\_0[t]]\ a\_1[a\_0[t], \[Phi]\_0[t]] - 
                      3\ Sin[3\ \[Phi]\_0[t]]\ a\_0[t]\ 
                        \[Phi]\_1[a\_0[t], \[Phi]\_0[t]])\))\)\)}]}], "}"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Next, we use the method of separation of variables to separate fast and \
slowly varying terms in the first-order equations, ",
  StyleBox["ampEq3[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phaseEq3[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  StyleBox[".",
    FontSlant->"Italic"],
  " The slowly varying parts yield"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(cond36a = {A\_1[a\_0[t]] -> 0, \[CapitalPhi]\_1[a\_0[t]] -> 0}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["whereas the fast varying parts yield", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order1Eq = {ampEq3[\([1]\)], phaseEq3[\([1]\)]} /. cond36a\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SubsuperscriptBox["a", "1", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
            "]"}], "==", 
          \(1\/8\ \((
              2\ Sin[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2 + 
                2\ Sin[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2)\)\)}], 
        ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", 
              TagBox[\((0, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
            "]"}], "==", 
          \(1\/8\ \((
              6\ Cos[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t] + 
                2\ Cos[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t])\)\)}]}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To determine the solutions of ",
  StyleBox["order1Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " using ",
  StyleBox["DSolve",
    FontWeight->"Bold"],
  ", we transform them from partial-differential to ordinary-differential \
equations, replace ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_0[t]\)]],
  " with ",
  Cell[BoxData[
      \(TraditionalForm\`s\)]],
  ", and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order1EqTransf = 
      \(\(order1Eq /. {a\_1 -> \((a\_1[#2]\ &)\), 
              \[Phi]\_1 -> \((\[Phi]\_1[#2]\ &)\)}\) /. f_[t] -> f\) /. 
        \[Phi]\_0 -> s\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
            SubsuperscriptBox["a", "1", "\[Prime]",
              MultilineFunction->None], "[", "s", "]"}], "==", 
          \(1\/8\ \((
              2\ Sin[s]\ a\_0\%2\ \[Alpha]\_2 + 
                2\ Sin[3\ s]\ a\_0\%2\ \[Alpha]\_2)\)\)}], ",", 
        RowBox[{
          RowBox[{
            SubsuperscriptBox["\[Phi]", "1", "\[Prime]",
              MultilineFunction->None], "[", "s", "]"}], "==", 
          \(1\/8\ \((
              6\ Cos[s]\ a\_0\ \[Alpha]\_2 + 2\ Cos[3\ s]\ a\_0\ \[Alpha]\_2)
              \)\)}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Instead of replacing ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_0[t]\)]],
  ", we could have symbolized it. Using ",
  StyleBox["DSolve,",
    FontWeight->"Bold"],
  " we find that ",
  Cell[BoxData[
      \(TraditionalForm\`a\_1\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_1\)]],
  " are given by"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(rule36[1] = 
      \(\(\(DSolve[order1EqTransf[\([1]\)], a\_1[s], s]\)[\([1]\)] /. 
            C[_] -> 0\) /. {s -> \[Phi]\_0[t], a\_0 -> a\_0[t]}\) /. 
        a\_1[arg_] -> a\_1[a\_0[t], arg]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({a\_1[a\_0[t], \[Phi]\_0[t]] \[Rule] 
        1\/12\ \((
            \(-3\)\ Cos[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2 - 
              Cos[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(rule36[2] = 
      \(\(\(DSolve[order1EqTransf[\([2]\)], \[Phi]\_1[s], s]\)[\([1]\)] /. 
            C[_] -> 0\) /. {s -> \[Phi]\_0[t], a\_0 -> a\_0[t]}\) /. 
        \[Phi]\_1[arg_] -> \[Phi]\_1[a\_0[t], arg]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Phi]\_1[a\_0[t], \[Phi]\_0[t]] \[Rule] 
        1\/12\ \((
            9\ Sin[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t] + 
              Sin[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t])\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["which can be combined into", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(cond36b = Join[rule36[1], rule36[2]]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the slow- and fast- varying components of the first-order \
solution, ",
  StyleBox["cond36a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["cond36b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", into the second-order equations, we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2Eq = 
      \({ampEq3[\([2]\)], phaseEq3[\([2]\)]} /. cond36a\) /. cond36b\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{\(A\_2[a\_0[t]]\), "+", 
            RowBox[{
              SubsuperscriptBox["a", "2", 
                TagBox[\((0, 1)\),
                  Derivative],
                MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
              "]"}]}], "==", 
          \(1\/8\ \((
              2\ Sin[2\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^3 + 
                Sin[4\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^3 + 
                2\ \[Alpha]\_2\ 
                  \((1\/12\ Cos[\[Phi]\_0[t]]\ a\_0[t]\^2\ 
                        \((9\ Sin[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t] + 
                            Sin[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t])\) + 
                      1\/6\ Sin[\[Phi]\_0[t]]\ a\_0[t]\ 
                        \((\(-3\)\ Cos[\[Phi]\_0[t]]\ \[Alpha]\_2\ 
                              a\_0[t]\^2 - 
                            Cos[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2)\))
                    \) + 2\ \[Alpha]\_2\ 
                  \((1\/4\ Cos[3\ \[Phi]\_0[t]]\ a\_0[t]\^2\ 
                        \((9\ Sin[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t] + 
                            Sin[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t])\) + 
                      1\/6\ Sin[3\ \[Phi]\_0[t]]\ a\_0[t]\ 
                        \((\(-3\)\ Cos[\[Phi]\_0[t]]\ \[Alpha]\_2\ 
                              a\_0[t]\^2 - 
                            Cos[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2)\))
                    \))\)\)}], ",", 
        RowBox[{
          RowBox[{\(\[CapitalPhi]\_2[a\_0[t]]\), "+", 
            RowBox[{
              SubsuperscriptBox["\[Phi]", "2", 
                TagBox[\((0, 1)\),
                  Derivative],
                MultilineFunction->None], "[", \(a\_0[t], \[Phi]\_0[t]\), 
              "]"}]}], "==", 
          \(1\/8\ \((
              3\ \[Alpha]\_3\ a\_0[t]\^2 + 
                4\ Cos[2\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^2 + 
                Cos[4\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^2 + 
                6\ \[Alpha]\_2\ 
                  \((\(-\(1\/12\)\)\ Sin[\[Phi]\_0[t]]\ a\_0[t]\ 
                        \((9\ Sin[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t] + 
                            Sin[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t])\) + 
                      1\/12\ Cos[\[Phi]\_0[t]]\ 
                        \((\(-3\)\ Cos[\[Phi]\_0[t]]\ \[Alpha]\_2\ 
                              a\_0[t]\^2 - 
                            Cos[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2)\))
                    \) + 2\ \[Alpha]\_2\ 
                  \((\(-\(1\/4\)\)\ Sin[3\ \[Phi]\_0[t]]\ a\_0[t]\ 
                        \((9\ Sin[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t] + 
                            Sin[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t])\) + 
                      1\/12\ Cos[3\ \[Phi]\_0[t]]\ 
                        \((\(-3\)\ Cos[\[Phi]\_0[t]]\ \[Alpha]\_2\ 
                              a\_0[t]\^2 - 
                            Cos[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2)\))
                    \))\)\)}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Next, we use trigonometric identities to expand the right-hand sides of the \
second-order equations in Fourier series and obtain\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2Eqrhs = 
      \(\((\(#[\([2]\)] // TrigReduce\) // Expand)\)\ &\)\ /@\ order2Eq\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\(-\(9\/32\)\)\ Sin[2\ \[Phi]\_0[t]]\ \[Alpha]\_2\%2\ a\_0[t]\^3 + 
        5\/24\ Sin[4\ \[Phi]\_0[t]]\ \[Alpha]\_2\%2\ a\_0[t]\^3 + 
        1\/96\ Sin[6\ \[Phi]\_0[t]]\ \[Alpha]\_2\%2\ a\_0[t]\^3 + 
        1\/4\ Sin[2\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^3 + 
        1\/8\ Sin[4\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^3, 
      \(-\(5\/12\)\)\ \[Alpha]\_2\%2\ a\_0[t]\^2 - 
        3\/16\ Cos[2\ \[Phi]\_0[t]]\ \[Alpha]\_2\%2\ a\_0[t]\^2 + 
        1\/4\ Cos[4\ \[Phi]\_0[t]]\ \[Alpha]\_2\%2\ a\_0[t]\^2 + 
        1\/48\ Cos[6\ \[Phi]\_0[t]]\ \[Alpha]\_2\%2\ a\_0[t]\^2 + 
        3\/8\ \[Alpha]\_3\ a\_0[t]\^2 + 
        1\/2\ Cos[2\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^2 + 
        1\/8\ Cos[4\ \[Phi]\_0[t]]\ \[Alpha]\_3\ a\_0[t]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Since we are seeking an expansion valid to ",
  Cell[BoxData[
      \(TraditionalForm\`O(\[Epsilon]\^2)\)]],
  ", we do not need to solve for ",
  Cell[BoxData[
      \(TraditionalForm\`a\_2\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_2\)]],
  ". All we need to do is to investigate the above expressions to determine \
the slowly varying parts and determine ",
  Cell[BoxData[
      \(TraditionalForm\`A\_2\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPhi]\_2\)]],
  ". The result is"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(cond36c = {A\_2[a\_0[t]], \[CapitalPhi]\_2[a\_0[t]]} -> 
          \((order2Eqrhs /. {Sin[_] -> 0, Cos[_] -> 0})\) // Thread\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({A\_2[a\_0[t]] \[Rule] 0, 
      \[CapitalPhi]\_2[a\_0[t]] \[Rule] 
        \(-\(5\/12\)\)\ \[Alpha]\_2\%2\ a\_0[t]\^2 + 
          3\/8\ \[Alpha]\_3\ a\_0[t]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting the fast-varying components of the first-order solution, ",
  StyleBox["cond36b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", into the expansions for ",
  Cell[BoxData[
      \(TraditionalForm\`a\)]],
  " and \[Phi] and discarding terms of order higher than \[Epsilon], we \
obtain, to the second approximation, that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(asolF = \(asol /. cond36b\) /. \[Epsilon]\^2 -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(a[t] == 
      a\_0[t] + 
        1\/12\ \[Epsilon]\ 
          \((\(-3\)\ Cos[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2 - 
              Cos[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2)\)\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phisolF = \(phisol /. cond36b\) /. \[Epsilon]\^2 -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\[Phi][t] == 
      1\/12\ \[Epsilon]\ 
          \((9\ Sin[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t] + 
              Sin[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t])\) + 
        \[Phi]\_0[t]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting the solw-varying components of the first- and second-order \
solutions, ",
  StyleBox["cond36a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["cond36c",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", into ",
  StyleBox["a0Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phi0Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(a0Eq1 = \(a0Eq /. cond36a\) /. cond36c\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SubsuperscriptBox["a", "0", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phi0Eq1 = \(phi0Eq /. cond36a\) /. cond36c\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SubsuperscriptBox["\[Phi]", "0", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(1 + \[Epsilon]\^2\ 
          \((\(-\(5\/12\)\)\ \[Alpha]\_2\%2\ a\_0[t]\^2 + 
              3\/8\ \[Alpha]\_3\ a\_0[t]\^2)\)\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "It follows from the above equations that ",
  Cell[BoxData[
      \(TraditionalForm\`\(a\_0 = \)\)]],
  " constant and"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(phi0Rule = 
      \(DSolve[phi0Eq1 /. a\_0[t] -> a\_0, \[Phi]\_0[t], t]\)[\([1]\)] /. 
        C[1] -> \[Beta]\_0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Phi]\_0[t] \[Rule] 
        t\ \((1 - 5\/12\ \[Epsilon]\^2\ a\_0\%2\ \[Alpha]\_2\%2 + 
                3\/8\ \[Epsilon]\^2\ a\_0\%2\ \[Alpha]\_3)\) + \[Beta]\_0}
      \)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "where ",
  Cell[BoxData[
      \(TraditionalForm\`\[Beta]\_0\)]],
  " is a constant. Substituting these expansions into ",
  StyleBox["eq35a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain the following second-order expansion:"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq36a = 
      u[t] == \[Epsilon]\ a[t]\ Cos[\[Phi][t]] /. 
        \(({asolF, phisolF} /. Equal -> Rule)\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u[t] == 
      \[Epsilon]\ 
        Cos[1\/12\ \[Epsilon]\ 
              \((9\ Sin[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t] + 
                  Sin[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t])\) + 
            \[Phi]\_0[t]]\ 
        \((a\_0[t] + 
            1\/12\ \[Epsilon]\ 
              \((\(-3\)\ Cos[\[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2 - 
                  Cos[3\ \[Phi]\_0[t]]\ \[Alpha]\_2\ a\_0[t]\^2)\))\)\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "To compare the present solution with those obtained by using the method of \
multiple scales and the Lindstedt-Poincar\[EAcute] technique, we expand the \
circular functions in ",
  StyleBox["eq36a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " for small \[Epsilon] about ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi]\_0\)]],
  " and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(u[t] == 
      \((\(\(Series[eq36a[\([2]\)], {\[Epsilon], 0, 2}] // Normal\) // 
            TrigReduce\) // Expand)\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u[t] == 
      \[Epsilon]\ Cos[\[Phi]\_0[t]]\ a\_0[t] - 
        1\/2\ \[Epsilon]\^2\ \[Alpha]\_2\ a\_0[t]\^2 + 
        1\/6\ \[Epsilon]\^2\ Cos[2\ \[Phi]\_0[t]]\ \[Alpha]\_2\ 
          a\_0[t]\^2\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "This expansion and ",
  StyleBox["phi0Rule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " are in full agreement with those obtained by using the method of multiple \
scales."
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["3.7 The Krylov-Bogoliubov-Mitropolsky Technique", "Section",
  CellMargins->{{Inherited, 17.4375}, {Inherited, Inherited}},
  Evaluatable->False,
  TextAlignment->Left,
  TextJustification->1,
  AspectRatioFixed->True,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],

Cell[TextData[{
  "In this section, we describe a variant of the generalized method of \
averaging, namely the Krylov-Bogoliubov-Mitropolsky technique. It is often \
referred to as the ",
  StyleBox["asymptotic method",
    FontColor->RGBColor[0, 0, 1]],
  ". When the nonlinear terms are neglected, the solution of ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " is"
}], "Text",
  CellMargins->{{12, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(linearSol = u == \[Epsilon]\ a\ Cos[t + \[Beta]]; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "where ",
  StyleBox["a",
    FontSlant->"Italic"],
  " and \[Beta] are constants and \[Epsilon] is a small dimensionless \
parameter that is a measure of the amplitude. When the nonlinear terms are \
included, we consider ",
  StyleBox["linearSol",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " to be the first term in an approximate solution of ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " but with slowly varying rather than constant ",
  StyleBox["a",
    FontSlant->"Italic"],
  " and \[Beta]. Moreover, we introduce the fast scale",
  StyleBox[" ",
    FontSlant->"Italic"],
  Cell[BoxData[
      \(TraditionalForm\`\[Phi] = t + \[Beta]\)]],
  " and use ",
  Cell[BoxData[
      \(TraditionalForm\`a\)]],
  " to represent the slow variations. Thus, we seek a second-order uniform \
expansion of the solution of ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in the form"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(uSol = 
      u[a[t], \[Phi][t]] == 
        \[Epsilon]\ a[t]\ Cos[\[Phi][t]] + 
          Sum[\[Epsilon]\^i\ u\_i[a[t], \[Phi][t]], {i, 2, 3}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(u[a[t], \[Phi][t]] == 
      \[Epsilon]\ a[t]\ Cos[\[Phi][t]] + 
        \[Epsilon]\^2\ u\_2[a[t], \[Phi][t]] + 
        \[Epsilon]\^3\ u\_3[a[t], \[Phi][t]]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Again, terms of order ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^3\)]],
  " have been included in the expansion, but these terms will not be included \
in the final result. Since ",
  StyleBox["a",
    FontSlant->"Italic"],
  " and \[Beta] are slowly varying functions of ",
  StyleBox["t, ",
    FontSlant->"Italic"],
  "we express them in power series of \[Epsilon] in terms of ",
  Cell[BoxData[
      \(TraditionalForm\`a\)]],
  " as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(apEq = \(a'\)[t] == Sum[\[Epsilon]\^i\ A\_i[a[t]], {i, 2}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["a", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(\[Epsilon]\ A\_1[a[t]] + \[Epsilon]\^2\ A\_2[a[t]]\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phipEq = 
      \(\[Phi]'\)[t] == 
        1 + Sum[\[Epsilon]\^i\ \[CapitalPhi]\_i[a[t]], {i, 2}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["\[Phi]", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(1 + \[Epsilon]\ \[CapitalPhi]\_1[a[t]] + 
        \[Epsilon]\^2\ \[CapitalPhi]\_2[a[t]]\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "In what follows, we need ",
  Cell[BoxData[
      \(TraditionalForm\`\(a\^\[DoublePrime]\)[t]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Phi]\^\[DoublePrime]\)[t]\)]],
  ". To this end, we differentiate ",
  StyleBox["apEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phipEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " with respect to ",
  Cell[BoxData[
      \(TraditionalForm\`t\)]],
  ", use these equations to express ",
  Cell[BoxData[
      \(TraditionalForm\`\(a\^\[Prime]\)[t]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Phi]\^\[Prime]\)[t]\)]],
  " in terms of the ",
  Cell[BoxData[
      \(TraditionalForm\`A\_i[a[t]]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalPhi]\_i[a[t]]\)]],
  ", discard terms of order higher than ",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^2\)]],
  ", and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(appEq = 
      \((D[apEq, t] /. \((apEq /. Equal -> Rule)\) // ExpandAll)\) /. 
        \[Epsilon]\^\(n_ /; n > 2\) -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["a", "\[DoublePrime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      RowBox[{\(\[Epsilon]\^2\), " ", \(A\_1[a[t]]\), " ", 
        RowBox[{
          SubsuperscriptBox["A", "1", "\[Prime]",
            MultilineFunction->None], "[", \(a[t]\), "]"}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phippEq = 
      \((D[phipEq, t] /. \((apEq /. Equal -> Rule)\) // ExpandAll)\) /. 
        \[Epsilon]\^\(n_ /; n > 2\) -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["\[Phi]", "\[DoublePrime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      RowBox[{\(\[Epsilon]\^2\), " ", \(A\_1[a[t]]\), " ", 
        RowBox[{
          SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
            MultilineFunction->None], "[", \(a[t]\), "]"}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Thus, this method can be viewed as a multiple scales procedure with ",
  StyleBox["a",
    FontSlant->"Italic"],
  " and \[Phi] being the scales."
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the assumed expansion for ",
  Cell[BoxData[
      \(TraditionalForm\`u[t]\)]],
  ", expressed in function form, and the expansions for the derivatives of ",
  Cell[BoxData[
      \(TraditionalForm\`a[t]\)]],
  " and ",
  Cell[BoxData[
      \(TraditionalForm\`\[Phi][t]\)]],
  " into ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and keeping terms up to order \
",
  Cell[BoxData[
      \(TraditionalForm\`\[Epsilon]\^3\)]],
  ", we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq37a = 
      \((\(eq31g /. u -> Function[t, uSol[\([2]\)] // Evaluate]\) /. 
              \(({apEq, phipEq, appEq, phippEq} /. Equal -> Rule)\) // 
            ExpandAll)\) /. \[Epsilon]\^\(n_ /; n > 3\) -> 0\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(\[Epsilon]\^2\ a[t]\^2\ Cos[\[Phi][t]]\^2\ \[Alpha]\_2\), "+", 
        \(\[Epsilon]\^3\ a[t]\^3\ Cos[\[Phi][t]]\^3\ \[Alpha]\_3\), "-", 
        \(2\ \[Epsilon]\^2\ Sin[\[Phi][t]]\ A\_1[a[t]]\), "-", 
        \(2\ \[Epsilon]\^3\ Sin[\[Phi][t]]\ A\_2[a[t]]\), "+", 
        \(\[Epsilon]\^2\ u\_2[a[t], \[Phi][t]]\), "+", 
        \(2\ \[Epsilon]\^3\ a[t]\ Cos[\[Phi][t]]\ \[Alpha]\_2\ 
          u\_2[a[t], \[Phi][t]]\), "+", 
        \(\[Epsilon]\^3\ u\_3[a[t], \[Phi][t]]\), "-", 
        \(2\ \[Epsilon]\^2\ a[t]\ Cos[\[Phi][t]]\ \[CapitalPhi]\_1[a[t]]\), 
        "-", \(2\ \[Epsilon]\^3\ Sin[\[Phi][t]]\ A\_1[a[t]]\ 
          \[CapitalPhi]\_1[a[t]]\), "-", 
        \(\[Epsilon]\^3\ a[t]\ Cos[\[Phi][t]]\ \[CapitalPhi]\_1[a[t]]\^2\), 
        "-", \(2\ \[Epsilon]\^3\ a[t]\ Cos[\[Phi][t]]\ 
          \[CapitalPhi]\_2[a[t]]\), "+", 
        RowBox[{
        \(\[Epsilon]\^3\), " ", \(Cos[\[Phi][t]]\), " ", \(A\_1[a[t]]\), 
          " ", 
          RowBox[{
            SubsuperscriptBox["A", "1", "\[Prime]",
              MultilineFunction->None], "[", \(a[t]\), "]"}]}], "-", 
        RowBox[{
        \(\[Epsilon]\^3\), " ", \(a[t]\), " ", \(Sin[\[Phi][t]]\), " ", 
          \(A\_1[a[t]]\), " ", 
          RowBox[{
            SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
              MultilineFunction->None], "[", \(a[t]\), "]"}]}], "+", 
        RowBox[{\(\[Epsilon]\^2\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "2", 
              TagBox[\((0, 2)\),
                Derivative],
              MultilineFunction->None], "[", \(a[t], \[Phi][t]\), "]"}]}], 
        "+", 
        RowBox[{
        "2", " ", \(\[Epsilon]\^3\), " ", \(\[CapitalPhi]\_1[a[t]]\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "2", 
              TagBox[\((0, 2)\),
                Derivative],
              MultilineFunction->None], "[", \(a[t], \[Phi][t]\), "]"}]}], 
        "+", 
        RowBox[{\(\[Epsilon]\^3\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "3", 
              TagBox[\((0, 2)\),
                Derivative],
              MultilineFunction->None], "[", \(a[t], \[Phi][t]\), "]"}]}], 
        "+", 
        RowBox[{"2", " ", \(\[Epsilon]\^3\), " ", \(A\_1[a[t]]\), " ", 
          RowBox[{
            SubsuperscriptBox["u", "2", 
              TagBox[\((1, 1)\),
                Derivative],
              MultilineFunction->None], "[", \(a[t], \[Phi][t]\), "]"}]}]}], 
      "==", "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Equating coefficients of like powers of \[Epsilon] in ",
  StyleBox["eq37a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqEps = 
      Table[Coefficient[eq37a[\([1]\)], \[Epsilon]\^i] == 0, {i, 2, 3}] /. 
        f_[t] -> f\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
          \(a\^2\ Cos[\[Phi]]\^2\ \[Alpha]\_2\), "-", 
            \(2\ Sin[\[Phi]]\ A\_1[a]\), "+", \(u\_2[a, \[Phi]]\), "-", 
            \(2\ a\ Cos[\[Phi]]\ \[CapitalPhi]\_1[a]\), "+", 
            RowBox[{
              SubsuperscriptBox["u", "2", 
                TagBox[\((0, 2)\),
                  Derivative],
                MultilineFunction->None], "[", \(a, \[Phi]\), "]"}]}], "==", 
          "0"}], ",", 
        RowBox[{
          RowBox[{
          \(a\^3\ Cos[\[Phi]]\^3\ \[Alpha]\_3\), "-", 
            \(2\ Sin[\[Phi]]\ A\_2[a]\), "+", 
            \(2\ a\ Cos[\[Phi]]\ \[Alpha]\_2\ u\_2[a, \[Phi]]\), "+", 
            \(u\_3[a, \[Phi]]\), "-", 
            \(2\ Sin[\[Phi]]\ A\_1[a]\ \[CapitalPhi]\_1[a]\), "-", 
            \(a\ Cos[\[Phi]]\ \[CapitalPhi]\_1[a]\^2\), "-", 
            \(2\ a\ Cos[\[Phi]]\ \[CapitalPhi]\_2[a]\), "+", 
            RowBox[{\(Cos[\[Phi]]\), " ", \(A\_1[a]\), " ", 
              RowBox[{
                SubsuperscriptBox["A", "1", "\[Prime]",
                  MultilineFunction->None], "[", "a", "]"}]}], "-", 
            RowBox[{"a", " ", \(Sin[\[Phi]]\), " ", \(A\_1[a]\), " ", 
              RowBox[{
                SubsuperscriptBox["\[CapitalPhi]", "1", "\[Prime]",
                  MultilineFunction->None], "[", "a", "]"}]}], "+", 
            RowBox[{"2", " ", \(\[CapitalPhi]\_1[a]\), " ", 
              RowBox[{
                SubsuperscriptBox["u", "2", 
                  TagBox[\((0, 2)\),
                    Derivative],
                  MultilineFunction->None], "[", \(a, \[Phi]\), "]"}]}], 
            "+", 
            RowBox[{
              SubsuperscriptBox["u", "3", 
                TagBox[\((0, 2)\),
                  Derivative],
                MultilineFunction->None], "[", \(a, \[Phi]\), "]"}], "+", 
            RowBox[{"2", " ", \(A\_1[a]\), " ", 
              RowBox[{
                SubsuperscriptBox["u", "2", 
                  TagBox[\((1, 1)\),
                    Derivative],
                  MultilineFunction->None], "[", \(a, \[Phi]\), "]"}]}]}], 
          "==", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Next, we use trigonometric identities to expand the nonhomogeneous terms \
in ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " in a Fourier series and obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order2Eq = TrigReduce\ /@\ eqEps[\([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{\(1\/2\), " ", 
        RowBox[{"(", 
          RowBox[{
          \(a\^2\ \[Alpha]\_2\), "+", \(a\^2\ Cos[2\ \[Phi]]\ \[Alpha]\_2\), 
            "-", \(4\ Sin[\[Phi]]\ A\_1[a]\), "+", \(2\ u\_2[a, \[Phi]]\), 
            "-", \(4\ a\ Cos[\[Phi]]\ \[CapitalPhi]\_1[a]\), "+", 
            RowBox[{"2", " ", 
              RowBox[{
                SubsuperscriptBox["u", "2", 
                  TagBox[\((0, 2)\),
                    Derivative],
                  MultilineFunction->None], "[", \(a, \[Phi]\), "]"}]}]}], 
          ")"}]}], "==", "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Eliminating the terms that produce secular terms in ",
  Cell[BoxData[
      \(TraditionalForm\`u\_2\)]],
  " from ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " demands that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(rule37a = 
      \(Solve[Coefficient[order2Eq[\([1]\)], {Cos[\[Phi]], Sin[\[Phi]]}] == 
              0 // Thread, {\[CapitalPhi]\_1[a], A\_1[a]}]\)[\([1]\)]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[CapitalPhi]\_1[a] \[Rule] 0, A\_1[a] \[Rule] 0}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Then, the solution of the resulting ",
  StyleBox["order2Eq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " can be written as"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(rule37b = 
      \(DSolve[order2Eq /. rule37a, u\_2[a, \[Phi]], {a, \[Phi]}]\)[
            \([1]\)] /. C[_] -> \((0\ &)\) // Simplify\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_2[a, \[Phi]] \[Rule] 
        1\/6\ a\^2\ \((\(-3\) + Cos[2\ \[Phi]])\)\ \[Alpha]\_2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["which can be expressed in function form as", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(rule37c = {
        u\_2 -> Function[{a, \[Phi]}, rule37b[\([1, 2]\)] // Evaluate]}\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({u\_2 \[Rule] 
        Function[{a, \[Phi]}, 
          1\/6\ a\^2\ \((\(-3\) + Cos[2\ \[Phi]])\)\ \[Alpha]\_2]}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting ",
  StyleBox["rule37a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["rule37c",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["eqEps[[2]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq37b = \(eqEps[\([2]\)] /. rule37a\) /. rule37c\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(1\/3\ a\^3\ Cos[\[Phi]]\ \((\(-3\) + Cos[2\ \[Phi]])\)\ 
          \[Alpha]\_2\%2\), "+", \(a\^3\ Cos[\[Phi]]\^3\ \[Alpha]\_3\), "-", 
        \(2\ Sin[\[Phi]]\ A\_2[a]\), "+", \(u\_3[a, \[Phi]]\), "-", 
        \(2\ a\ Cos[\[Phi]]\ \[CapitalPhi]\_2[a]\), "+", 
        RowBox[{
          SubsuperscriptBox["u", "3", 
            TagBox[\((0, 2)\),
              Derivative],
            MultilineFunction->None], "[", \(a, \[Phi]\), "]"}]}], "==", 
      "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Eliminating the terms that produce secular terms in ",
  Cell[BoxData[
      \(TraditionalForm\`u\_3\)]],
  " from ",
  StyleBox["eq37b",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " demands that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eq37c = 
      Coefficient[
            eq37b[\([1]\)] // TrigReduce, {Cos[\[Phi]], Sin[\[Phi]]}] == 0 // 
        Thread\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\(-\(5\/6\)\)\ a\^3\ \[Alpha]\_2\%2 + \(3\ a\^3\ \[Alpha]\_3\)\/4 - 
          2\ a\ \[CapitalPhi]\_2[a] == 0, \(-2\)\ A\_2[a] == 0}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["or ", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(rule37d = \(Solve[eq37c, {A\_2[a], \[CapitalPhi]\_2[a]}]\)[\([1]\)]\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({A\_2[a] \[Rule] 0, 
      \[CapitalPhi]\_2[a] \[Rule] 
        \(-\(1\/24\)\)\ a\^2\ \((10\ \[Alpha]\_2\%2 - 9\ \[Alpha]\_3)\)}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting ",
  StyleBox["rule37a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["rule37d",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " into ",
  StyleBox["apEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " and ",
  StyleBox["phipEq",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(apEq /. \((rule37a~Join~rule37d /. a -> a[t])\)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["a", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", "0"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
    \(phipEq /. \((rule37a~Join~rule37d /. a -> a[t])\) // ExpandAll\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
        SuperscriptBox["\[Phi]", "\[Prime]",
          MultilineFunction->None], "[", "t", "]"}], "==", 
      \(1 - 5\/12\ \[Epsilon]\^2\ a[t]\^2\ \[Alpha]\_2\%2 + 
        3\/8\ \[Epsilon]\^2\ a[t]\^2\ \[Alpha]\_3\)}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
which are in full agreement with those obtained by using the method of \
multiple scales and the generalized method of averaging.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["3.8 The Method of Normal Forms", "Section",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "In this section, we use the method of normal forms to determine a \
second-order uniform expansion of the solution of ",
  StyleBox["eq31g",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ". To this end, we start with the corresponding first-order complex-valued \
equation"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{"eq343a", "=", 
        RowBox[{
          RowBox[{
            SuperscriptBox["\[Zeta]", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}], "==", 
          \(I\ \[Zeta][t] + 1\/2\ I\ \[Alpha]\_2\ \[Zeta][t]\^2 + 
            1\/2\ I\ \[Alpha]\_3\ \[Zeta][t]\^3 + 
            I\ \[Alpha]\_2\ \[Zeta][t]\ \[Zeta]\&_[t] + 
            3\/2\ I\ \[Alpha]\_3\ \[Zeta][t]\^2\ \[Zeta]\&_[t] + 
            1\/2\ I\ \[Alpha]\_2\ \[Zeta]\&_[t]\^2 + 
            3\/2\ I\ \[Alpha]\_3\ \[Zeta][t]\ \[Zeta]\&_[t]\^2 + 
            1\/2\ I\ \[Alpha]\_3\ \[Zeta]\&_[t]\^3\)}]}], ";"}]], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
According to the method of normal forms, we introduce the near-identity \
transformation\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(basicTerms = {\[Eta][t], \[Eta]\&_[t]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(zetaRule = {\n
        \[Zeta] -> 
          Function[t, 
            \[Epsilon]\ \[Eta][t] + 
                Sum[\[Epsilon]\^j\ h\_j\ @@\ basicTerms, {j, 2, 3}] // 
              Evaluate], \n
        \[Zeta]\&_ -> 
          Function[t, 
            \[Epsilon]\ \[Eta]\&_[t] + 
                Sum[\[Epsilon]\^j\ h\&_\_j\ @@\ basicTerms, {j, 2, 3}] // 
              Evaluate]}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Zeta] \[Rule] 
        Function[t, 
          \[Epsilon]\ \[Eta][t] + 
            \[Epsilon]\^2\ h\_2[\[Eta][t], \[Eta]\&_[t]] + 
            \[Epsilon]\^3\ h\_3[\[Eta][t], \[Eta]\&_[t]]], 
      \[Zeta]\&_ \[Rule] 
        Function[t, 
          \[Epsilon]\ \[Eta]\&_[t] + 
            \[Epsilon]\^2\ h\&_\_2[\[Eta][t], \[Eta]\&_[t]] + 
            \[Epsilon]\^3\ h\&_\_3[\[Eta][t], \[Eta]\&_[t]]]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["that results in the simplest possible equation", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(etaRule = {
        \(\[Eta]\^\[Prime]\)[t] -> 
          I\ \[Eta][t] + Sum[\[Epsilon]\^j\ g\_j[t], {j, 2}]}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          SuperscriptBox["\[Eta]", "\[Prime]",
            MultilineFunction->None], "[", "t", "]"}], "\[Rule]", 
        \(I\ \[Eta][t] + \[Epsilon]\ g\_1[t] + \[Epsilon]\^2\ g\_2[t]\)}], 
      "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["where the overbar denotes the complex conjugate", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(conjugateRule = {\[Eta] -> \[Eta]\&_, \[Eta]\&_ -> \[Eta], g -> g\&_, 
        g\&_ -> g, Complex[0, n_] -> Complex[0, \(-n\)]}; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Substituting the expansion for ",
  Cell[BoxData[
      \(TraditionalForm\`\[Zeta]\)]],
  ", the ",
  StyleBox["zetaRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", into ",
  StyleBox["eq343a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", using ",
  StyleBox["etaRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", expanding the result for small \[Epsilon], and discarding terms of order \
higher than ",
  Cell[BoxData[
      \(TraditionalForm\`\(\[Epsilon]\^3, \)\)]],
  " we have"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(eq38a = 
      \((\(\(eq343a /. zetaRule\) /. etaRule\) /. 
              \((etaRule /. conjugateRule)\) // ExpandAll)\) /. 
        \[Epsilon]\^\(n_ /; n > 3\) -> 0; \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "Equating coefficients of like powers of \[Epsilon] in ",
  StyleBox["eq38a",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(eqEps = 
      Table[Coefficient[Subtract\ @@\ eq38a, \[Epsilon]\^i] == 0, {i, 2, 3}]
        \)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{"{", 
      RowBox[{
        RowBox[{
          RowBox[{
          \(\(-\(1\/2\)\)\ I\ \[Alpha]\_2\ \[Eta][t]\^2\), "-", 
            \(I\ \[Alpha]\_2\ \[Eta][t]\ \[Eta]\&_[t]\), "-", 
            \(1\/2\ I\ \[Alpha]\_2\ \[Eta]\&_[t]\^2\), "+", \(g\_1[t]\), "-", 
            \(I\ h\_2[\[Eta][t], \[Eta]\&_[t]]\), "-", 
            RowBox[{"I", " ", \(\[Eta]\&_[t]\), " ", 
              RowBox[{
                SubsuperscriptBox["h", "2", 
                  TagBox[\((0, 1)\),
                    Derivative],
                  MultilineFunction->None], "[", \(\[Eta][t], \[Eta]\&_[t]\), 
                "]"}]}], "+", 
            RowBox[{"I", " ", \(\[Eta][t]\), " ", 
              RowBox[{
                SubsuperscriptBox["h", "2", 
                  TagBox[\((1, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(\[Eta][t], \[Eta]\&_[t]\), 
                "]"}]}]}], "==", "0"}], ",", 
        RowBox[{
          RowBox[{
          \(\(-\(1\/2\)\)\ I\ \[Alpha]\_3\ \[Eta][t]\^3\), "-", 
            \(3\/2\ I\ \[Alpha]\_3\ \[Eta][t]\^2\ \[Eta]\&_[t]\), "-", 
            \(3\/2\ I\ \[Alpha]\_3\ \[Eta][t]\ \[Eta]\&_[t]\^2\), "-", 
            \(1\/2\ I\ \[Alpha]\_3\ \[Eta]\&_[t]\^3\), "+", \(g\_2[t]\), "-", 
            \(I\ \[Alpha]\_2\ \[Eta][t]\ h\_2[\[Eta][t], \[Eta]\&_[t]]\), 
            "-", \(I\ \[Alpha]\_2\ \[Eta]\&_[t]\ 
              h\_2[\[Eta][t], \[Eta]\&_[t]]\), "-", 
            \(I\ h\_3[\[Eta][t], \[Eta]\&_[t]]\), "-", 
            \(I\ \[Alpha]\_2\ \[Eta][t]\ h\&_\_2[\[Eta][t], \[Eta]\&_[t]]\), 
            "-", \(I\ \[Alpha]\_2\ \[Eta]\&_[t]\ 
              h\&_\_2[\[Eta][t], \[Eta]\&_[t]]\), "+", 
            RowBox[{\(g\&_\_1[t]\), " ", 
              RowBox[{
                SubsuperscriptBox["h", "2", 
                  TagBox[\((0, 1)\),
                    Derivative],
                  MultilineFunction->None], "[", \(\[Eta][t], \[Eta]\&_[t]\), 
                "]"}]}], "-", 
            RowBox[{"I", " ", \(\[Eta]\&_[t]\), " ", 
              RowBox[{
                SubsuperscriptBox["h", "3", 
                  TagBox[\((0, 1)\),
                    Derivative],
                  MultilineFunction->None], "[", \(\[Eta][t], \[Eta]\&_[t]\), 
                "]"}]}], "+", 
            RowBox[{\(g\_1[t]\), " ", 
              RowBox[{
                SubsuperscriptBox["h", "2", 
                  TagBox[\((1, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(\[Eta][t], \[Eta]\&_[t]\), 
                "]"}]}], "+", 
            RowBox[{"I", " ", \(\[Eta][t]\), " ", 
              RowBox[{
                SubsuperscriptBox["h", "3", 
                  TagBox[\((1, 0)\),
                    Derivative],
                  MultilineFunction->None], "[", \(\[Eta][t], \[Eta]\&_[t]\), 
                "]"}]}]}], "==", "0"}]}], "}"}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["Second-Order Solution", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell[TextData[{
  "We choose the ",
  Cell[BoxData[
      \(TraditionalForm\`h\_i\)]],
  " to eliminate as many terms from ",
  StyleBox["eqEps",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", thereby reducing them into their simplest possible form. It turns out \
that we can eliminate all nonresonance terms. To determine the resonance \
terms in ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", we note that all of the possible forms of its nonhomogeneous terms are"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(possibleQTerms = 
      \(Outer[Times, basicTerms, basicTerms] // Flatten\) // Union\)], "Input",\

  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Eta][t]\^2, \[Eta][t]\ \[Eta]\&_[t], \[Eta]\&_[t]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "It follows from the linear parts of the ",
  StyleBox["etaRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " that"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(\(form = {\[Eta][t] -> E\^\(I\ t\), \[Eta]\&_[t] -> E\^\(\(-I\)\ t\)}; 
    \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell["Hence, the possible resonance terms are given by", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ResonantQTerm = 
      \(\((\(\(E\^\(\(-I\)\ t\)\) possibleQTerms /. form\) /. 
                E\^\(_\ t\) -> 0)\)\ possibleQTerms // Union\) // Rest\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Consequently, there are no resonance terms in the second-order problem and \
the nonresonance terms can be defined as\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(NRQT = Complement[possibleQTerms, ResonantQTerm]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Eta][t]\^2, \[Eta][t]\ \[Eta]\&_[t], \[Eta]\&_[t]\^2}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["We associate with each of them a coefficient according to", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(coeffsQ = Table[\[CapitalGamma]\_j, {j, Length[NRQT]}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[CapitalGamma]\_1, \[CapitalGamma]\_2, \[CapitalGamma]\_3}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Therefore, ",
  Cell[BoxData[
      \(TraditionalForm\`h\_2\)]],
  " and its complex conjugate have the form"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(hFormQ = {
        h\_2 -> \((
            Evaluate[coeffsQ . NRQT /. Thread[basicTerms -> {#1, #2}]]\ &)\), 
        h\&_\_2 -> 
          \((Evaluate[
                \(coeffsQ . NRQT /. conjugateRule\) /. 
                  Thread[basicTerms -> {#1, #2}]]\ &)\)}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({h\_2 \[Rule] 
        \((#1\^2\ \[CapitalGamma]\_1 + #1\ #2\ \[CapitalGamma]\_2 + #2\^2\ 
                \[CapitalGamma]\_3&)\), 
      h\&_\_2 \[Rule] 
        \((#2\^2\ \[CapitalGamma]\_1 + #1\ #2\ \[CapitalGamma]\_2 + #1\^2\ 
                \[CapitalGamma]\_3&)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting for ",
  Cell[BoxData[
      \(TraditionalForm\`h\_2\)]],
  " in ",
  StyleBox["eqEps[[1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", equating the coefficients of the possible nonresonance terms to zero, \
and solving the resulting equations for the ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalGamma]\_i\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(coeffsQRule = 
      \(Solve[Coefficient[eqEps[\([1, 1]\)] /. hFormQ, NRQT] == 0 // Thread, 
          coeffsQ]\)[\([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[CapitalGamma]\_1 \[Rule] \[Alpha]\_2\/2, 
      \[CapitalGamma]\_2 \[Rule] \(-\[Alpha]\_2\), 
      \[CapitalGamma]\_3 \[Rule] \(-\(\[Alpha]\_2\/6\)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "We choose ",
  Cell[BoxData[
      \(TraditionalForm\`g\_1\)]],
  " to eliminate the resonance terms in ",
  StyleBox["eqEps[[1,1]]",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " according to"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(gRuleQ = 
      g\_1[t] -> 
        \(-\ Coefficient[eqEps[\([1, 1]\)], ResonantQTerm] . ResonantQTerm
            \)\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(g\_1[t] \[Rule] 0\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "In this case, there are on resonance terms and hence ",
  Cell[BoxData[
      \(TraditionalForm\`g\_1 = 0. \)]]
}], "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[CellGroupData[{

Cell["Third-Order Equations", "Subsection",
  TextAlignment->Left,
  TextJustification->1],

Cell["\<\
Substituting the second-order results into the third-order equation yields\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(order3expr = 
      \(\(\(eqEps[\([2, 1]\)] /. hFormQ\) /. coeffsQRule\) /. gRuleQ\) /. 
          \((gRuleQ /. conjugateRule)\) // Expand\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
    \(\(-\(1\/3\)\)\ I\ \[Alpha]\_2\%2\ \[Eta][t]\^3\), "-", 
      \(1\/2\ I\ \[Alpha]\_3\ \[Eta][t]\^3\), "+", 
      \(5\/3\ I\ \[Alpha]\_2\%2\ \[Eta][t]\^2\ \[Eta]\&_[t]\), "-", 
      \(3\/2\ I\ \[Alpha]\_3\ \[Eta][t]\^2\ \[Eta]\&_[t]\), "+", 
      \(5\/3\ I\ \[Alpha]\_2\%2\ \[Eta][t]\ \[Eta]\&_[t]\^2\), "-", 
      \(3\/2\ I\ \[Alpha]\_3\ \[Eta][t]\ \[Eta]\&_[t]\^2\), "-", 
      \(1\/3\ I\ \[Alpha]\_2\%2\ \[Eta]\&_[t]\^3\), "-", 
      \(1\/2\ I\ \[Alpha]\_3\ \[Eta]\&_[t]\^3\), "+", \(g\_2[t]\), "-", 
      \(I\ h\_3[\[Eta][t], \[Eta]\&_[t]]\), "-", 
      RowBox[{"I", " ", \(\[Eta]\&_[t]\), " ", 
        RowBox[{
          SubsuperscriptBox["h", "3", 
            TagBox[\((0, 1)\),
              Derivative],
            MultilineFunction->None], "[", \(\[Eta][t], \[Eta]\&_[t]\), 
          "]"}]}], "+", 
      RowBox[{"I", " ", \(\[Eta][t]\), " ", 
        RowBox[{
          SubsuperscriptBox["h", "3", 
            TagBox[\((1, 0)\),
              Derivative],
            MultilineFunction->None], "[", \(\[Eta][t], \[Eta]\&_[t]\), 
          "]"}]}]}]], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "The nonhomogeneous terms in ",
  StyleBox["order3expr",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " are proportional to"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(possibleCTerms = 
      \(Outer[Times, possibleQTerms, basicTerms] // Flatten\) // Union\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Eta][t]\^3, \[Eta][t]\^2\ \[Eta]\&_[t], \[Eta][t]\ \[Eta]\&_[t]\^2, 
      \[Eta]\&_[t]\^3}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["Next, we determine the resonance terms according to", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(ResonantCTerm = 
      \(\((\(\(E\^\(\(-I\)\ t\)\) possibleCTerms /. form\) /. 
                E\^\(_\ t\) -> 0)\)\ possibleCTerms // Union\) // Rest\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Eta][t]\^2\ \[Eta]\&_[t]}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Then, the nonresonance terms are the complement of these resonance terms; \
that is,\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(NRCT = Complement[possibleCTerms, ResonantCTerm]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[Eta][t]\^3, \[Eta][t]\ \[Eta]\&_[t]\^2, \[Eta]\&_[t]\^3}\)], "Output",\

  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
Again, we associate coefficients with these terms as follows:\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(coeffsC = Table[\[CapitalLambda]\_j, {j, Length[NRCT]}]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[CapitalLambda]\_1, \[CapitalLambda]\_2, \[CapitalLambda]\_3}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Consequently, ",
  Cell[BoxData[
      \(TraditionalForm\`h\_3\)]],
  " and its complex conjugate have the forms"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(hFormC = {
        h\_3 -> \((
            Evaluate[coeffsC . NRCT /. Thread[basicTerms -> {#1, #2}]]\ &)\), 
        h\&_\_3 -> 
          \((Evaluate[
                \(coeffsC . NRCT /. conjugateRule\) /. 
                  Thread[basicTerms -> {#1, #2}]]\ &)\)}\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({h\_3 \[Rule] 
        \((#1\^3\ \[CapitalLambda]\_1 + #1\ #2\^2\ 
                \[CapitalLambda]\_2 + #2\^3\ \[CapitalLambda]\_3&)\), 
      h\&_\_3 \[Rule] 
        \((#2\^3\ \[CapitalLambda]\_1 + #1\^2\ #2\ 
                \[CapitalLambda]\_2 + #1\^3\ \[CapitalLambda]\_3&)\)}\)], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Substituting for ",
  Cell[BoxData[
      \(TraditionalForm\`h\_3\)]],
  " into ",
  StyleBox["order3expr",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", equating the coefficient of each possible nonresonance term to zero, and \
solving the resulting algebraic equations for the ",
  Cell[BoxData[
      \(TraditionalForm\`\[CapitalLambda]\_i\)]],
  ", we obtain"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(coeffsCRule = 
      \(Solve[Coefficient[order3expr /. hFormC, NRCT] == 0 // Thread, coeffsC]
          \)[\([1]\)]\)], "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \({\[CapitalLambda]\_1 \[Rule] 
        1\/12\ \((2\ \[Alpha]\_2\%2 + 3\ \[Alpha]\_3)\), 
      \[CapitalLambda]\_2 \[Rule] 
        1\/12\ \((10\ \[Alpha]\_2\%2 - 9\ \[Alpha]\_3)\), 
      \[CapitalLambda]\_3 \[Rule] 
        1\/24\ \((\(-2\)\ \[Alpha]\_2\%2 - 3\ \[Alpha]\_3)\)}\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Choosing ",
  Cell[BoxData[
      \(TraditionalForm\`g\_2\)]],
  " to eliminate the resonance terms from ",
  StyleBox["order3expr",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  " yields"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(gRuleC = 
      g\_2[t] -> 
        \(-\ Coefficient[order3expr, ResonantCTerm] . ResonantCTerm\)\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    \(g\_2[t] \[Rule] 
      \(-\((\(5\ I\ \[Alpha]\_2\%2\)\/3 - \(3\ I\ \[Alpha]\_3\)\/2)\)\)\ 
        \[Eta][t]\^2\ \[Eta]\&_[t]\)], "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell[TextData[{
  "Combining ",
  StyleBox["etaRule",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", ",
  StyleBox["gRuleQ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  ", and ",
  StyleBox["gRuleC ",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 1]],
  "and letting ",
  Cell[BoxData[
      \(TraditionalForm\`\[Eta] = A[t]\ E\^\(I\ t\)\)]],
  ", we obtain the modulation equation"
}], "Text",
  TextAlignment->Left,
  TextJustification->1],

Cell[CellGroupData[{

Cell[BoxData[
    \(moduEq = 
      2\ I\ \(E\^\(\(-I\)\ t\)\) 
              \((\(\[Eta]\^\[Prime]\)[t] - 
                  \((\(\(\(\[Eta]\^\[Prime]\)[t] /. etaRule\) /. gRuleQ\) /. 
                      gRuleC)\))\) == 0 /. {
            \[Eta] -> \((A[#]\ E\^\(I\ #\)&)\), 
            \[Eta]\&_ -> \((A\&_[#]\ E\^\(\(-I\)\ #\)&)\)} // ExpandAll\)], 
  "Input",
  TextAlignment->Left,
  TextJustification->1],

Cell[BoxData[
    RowBox[{
      RowBox[{
      \(\(-\(10\/3\)\)\ \[Epsilon]\^2\ A[t]\^2\ \[Alpha]\_2\%2\ A\&_[t]\), 
        "+", \(3\ \[Epsilon]\^2\ A[t]\^2\ \[Alpha]\_3\ A\&_[t]\), "+", 
        RowBox[{"2", " ", "I", " ", 
          RowBox[{
            SuperscriptBox["A", "\[Prime]",
              MultilineFunction->None], "[", "t", "]"}]}]}], "==", "0"}]], 
  "Output",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]],

Cell["\<\
which is in full agreement with that obtained by using the other \
techniques.\
\>", "Text",
  TextAlignment->Left,
  TextJustification->1]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"Microsoft Windows 3.0",
ScreenRectangle->{{0, 1024}, {0, 712}},
WindowToolbars->{"RulerBar", "EditBar"},
WindowSize->{709, 564},
WindowMargins->{{3, Automatic}, {Automatic, 5}},
PrintingCopies->1,
PrintingStartingPageNumber->45,
PrintingPageRange->{45, 45},
PageHeaders->{{Cell[ 
        TextData[ {
          CounterBox[ "Page"]}], "PageNumber"], Inherited, Cell[ 
        TextData[ {
          ValueBox[ "FileName"]}], "Header"]}, {Cell[ 
        TextData[ {
          ValueBox[ "FileName"]}], "Header"], Inherited, Cell[ 
        TextData[ {
          CounterBox[ "Page"]}], "PageNumber"]}},
PrintingOptions->{"FirstPageHeader"->False,
"FacingPages"->True},
Magnification->1
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{
  "ChapterTwo"->{
    Cell[12035, 447, 268, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[22340, 810, 283, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[40203, 1411, 272, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[43804, 1530, 279, 6, 43, "Subsection",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[77793, 2678, 278, 6, 43, "Subsection",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[113922, 3875, 281, 6, 43, "Subsection",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[136939, 4650, 266, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[150123, 5123, 278, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[184896, 6259, 286, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}]},
  "ChapterTwo.2"->{
    Cell[12035, 447, 268, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[22340, 810, 283, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[40203, 1411, 272, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[43804, 1530, 279, 6, 43, "Subsection",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[77793, 2678, 278, 6, 43, "Subsection",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[113922, 3875, 281, 6, 43, "Subsection",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[136939, 4650, 266, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[150123, 5123, 278, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[184896, 6259, 286, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}]},
  "ChapterTwo.2.1"->{
    Cell[12035, 447, 268, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[22340, 810, 283, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[40203, 1411, 272, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[43804, 1530, 279, 6, 43, "Subsection",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[77793, 2678, 278, 6, 43, "Subsection",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[113922, 3875, 281, 6, 43, "Subsection",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[136939, 4650, 266, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[150123, 5123, 278, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
    Cell[184896, 6259, 286, 6, 46, "Section",
      Evaluatable->False,
      CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}]}
  }
*)

(*CellTagsIndex
CellTagsIndex->{
  {"ChapterTwo", 226842, 7712},
  {"ChapterTwo.2", 228116, 7740},
  {"ChapterTwo.2.1", 229392, 7768}
  }
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1731, 51, 129, 5, 195, "Title"],
Cell[1863, 58, 100, 3, 30, "Input"],

Cell[CellGroupData[{
Cell[1988, 65, 103, 2, 53, "Section"],
Cell[2094, 69, 277, 8, 52, "Text"],

Cell[CellGroupData[{
Cell[2396, 81, 175, 5, 30, "Input"],
Cell[2574, 88, 329, 9, 29, "Output"]
}, Open  ]],
Cell[2918, 100, 467, 16, 52, "Text"],

Cell[CellGroupData[{
Cell[3410, 120, 126, 3, 30, "Input"],
Cell[3539, 125, 386, 11, 29, "Output"]
}, Open  ]],
Cell[3940, 139, 260, 7, 33, "Text"],

Cell[CellGroupData[{
Cell[4225, 150, 129, 3, 30, "Input"],
Cell[4357, 155, 150, 4, 22, "Output"]
}, Open  ]],
Cell[4522, 162, 255, 6, 40, "Text"],
Cell[4780, 170, 145, 4, 22, "Input"],
Cell[4928, 176, 335, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[5288, 192, 177, 5, 22, "Input"],
Cell[5468, 199, 525, 13, 22, "Output"]
}, Open  ]],
Cell[6008, 215, 134, 4, 26, "Text"],
Cell[6145, 221, 167, 4, 22, "Input"],
Cell[6315, 227, 180, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[6520, 239, 181, 5, 22, "Input"],
Cell[6704, 246, 503, 12, 30, "Output"]
}, Open  ]],
Cell[7222, 261, 178, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[7425, 273, 194, 6, 22, "Input"],
Cell[7622, 281, 343, 10, 22, "Output"]
}, Open  ]],
Cell[7980, 294, 166, 7, 26, "Text"],

Cell[CellGroupData[{
Cell[8171, 305, 184, 5, 22, "Input"],
Cell[8358, 312, 334, 7, 30, "Output"]
}, Open  ]],
Cell[8707, 322, 254, 10, 26, "Text"],

Cell[CellGroupData[{
Cell[8986, 336, 137, 3, 22, "Input"],
Cell[9126, 341, 149, 4, 36, "Output"]
}, Open  ]],
Cell[9290, 348, 246, 10, 26, "Text"],

Cell[CellGroupData[{
Cell[9561, 362, 112, 3, 22, "Input"],
Cell[9676, 367, 347, 8, 34, "Output"]
}, Open  ]],
Cell[10038, 378, 255, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[10318, 393, 113, 3, 22, "Input"],
Cell[10434, 398, 330, 10, 22, "Output"]
}, Open  ]],
Cell[10779, 411, 322, 12, 26, "Text"],
Cell[11104, 425, 894, 17, 97, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[12035, 447, 268, 6, 46, "Section",
  Evaluatable->False,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
Cell[12306, 455, 398, 11, 40, "Text"],
Cell[12707, 468, 171, 5, 22, "Input"],
Cell[12881, 475, 725, 19, 69, "Text"],

Cell[CellGroupData[{
Cell[13631, 498, 179, 5, 22, "Input"],
Cell[13813, 505, 959, 23, 45, "Output"]
}, Open  ]],
Cell[14787, 531, 225, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[15037, 543, 160, 5, 22, "Input"],
Cell[15200, 550, 912, 24, 22, "Output"]
}, Open  ]],
Cell[16127, 577, 244, 10, 26, "Text"],
Cell[16374, 589, 240, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[16639, 601, 256, 6, 34, "Input"],
Cell[16898, 609, 118, 3, 22, "Output"]
}, Open  ]],
Cell[17031, 615, 615, 24, 40, "Text"],

Cell[CellGroupData[{
Cell[17671, 643, 183, 5, 22, "Input"],
Cell[17857, 650, 180, 5, 28, "Output"]
}, Open  ]],
Cell[18052, 658, 216, 8, 26, "Text"],
Cell[18271, 668, 493, 19, 26, "Text"],

Cell[CellGroupData[{
Cell[18789, 691, 197, 5, 22, "Input"],
Cell[18989, 698, 416, 9, 43, "Output"]
}, Open  ]],
Cell[19420, 710, 295, 8, 40, "Text"],

Cell[CellGroupData[{
Cell[19740, 722, 123, 3, 22, "Input"],
Cell[19866, 727, 341, 7, 43, "Output"]
}, Open  ]],
Cell[20222, 737, 174, 5, 26, "Text"],

Cell[CellGroupData[{
Cell[20421, 746, 172, 5, 22, "Input"],
Cell[20596, 753, 519, 11, 49, "Output"]
}, Open  ]],
Cell[21130, 767, 552, 20, 40, "Text"],
Cell[21685, 789, 618, 16, 54, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[22340, 810, 283, 6, 46, "Section",
  Evaluatable->False,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
Cell[22626, 818, 400, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[23051, 834, 163, 5, 22, "Input"],
Cell[23217, 841, 401, 11, 22, "Output"]
}, Open  ]],
Cell[23633, 855, 353, 11, 26, "Text"],
Cell[23989, 868, 277, 7, 34, "Input"],
Cell[24269, 877, 613, 12, 69, "Text"],
Cell[24885, 891, 165, 3, 26, "Text"],
Cell[25053, 896, 98, 3, 22, "Input"],
Cell[25154, 901, 492, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[25671, 921, 218, 5, 34, "Input"],
Cell[25892, 928, 1886, 39, 45, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[27815, 972, 160, 5, 22, "Input"],
Cell[27978, 979, 1835, 42, 34, "Output"]
}, Open  ]],
Cell[29828, 1024, 240, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[30093, 1036, 242, 6, 22, "Input"],
Cell[30338, 1044, 166, 5, 22, "Output"]
}, Open  ]],
Cell[30519, 1052, 539, 20, 40, "Text"],
Cell[31061, 1074, 462, 16, 40, "Text"],

Cell[CellGroupData[{
Cell[31548, 1094, 429, 11, 22, "Input"],
Cell[31980, 1107, 378, 9, 22, "Output"]
}, Open  ]],
Cell[32373, 1119, 255, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[32653, 1131, 125, 3, 22, "Input"],
Cell[32781, 1136, 236, 6, 28, "Output"]
}, Open  ]],
Cell[33032, 1145, 461, 17, 40, "Text"],

Cell[CellGroupData[{
Cell[33518, 1166, 198, 5, 22, "Input"],
Cell[33719, 1173, 194, 5, 28, "Output"]
}, Open  ]],
Cell[33928, 1181, 94, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[34047, 1187, 145, 4, 22, "Input"],
Cell[34195, 1193, 209, 6, 28, "Output"]
}, Open  ]],
Cell[34419, 1202, 493, 19, 26, "Text"],

Cell[CellGroupData[{
Cell[34937, 1225, 490, 13, 22, "Input"],
Cell[35430, 1240, 501, 11, 28, "Output"]
}, Open  ]],
Cell[35946, 1254, 257, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[36228, 1266, 125, 3, 22, "Input"],
Cell[36356, 1271, 389, 8, 41, "Output"]
}, Open  ]],
Cell[36760, 1282, 236, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[37021, 1294, 212, 5, 22, "Input"],
Cell[37236, 1301, 185, 5, 29, "Output"]
}, Open  ]],
Cell[37436, 1309, 329, 9, 40, "Text"],

Cell[CellGroupData[{
Cell[37790, 1322, 192, 5, 22, "Input"],
Cell[37985, 1329, 310, 7, 28, "Output"]
}, Open  ]],
Cell[38310, 1339, 68, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[38403, 1345, 162, 4, 22, "Input"],
Cell[38568, 1351, 245, 7, 29, "Output"]
}, Open  ]],
Cell[38828, 1361, 326, 8, 40, "Text"],
Cell[39157, 1371, 1009, 35, 55, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[40203, 1411, 272, 6, 46, "Section",
  Evaluatable->False,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
Cell[40478, 1419, 871, 25, 69, "Text"],
Cell[41352, 1446, 112, 3, 22, "Input"],
Cell[41467, 1451, 456, 15, 22, "Input"],
Cell[41926, 1468, 132, 3, 26, "Text"],
Cell[42061, 1473, 117, 3, 22, "Input"],
Cell[42181, 1478, 251, 8, 26, "Text"],
Cell[42435, 1488, 317, 7, 34, "Input"],
Cell[42755, 1497, 165, 3, 26, "Text"],
Cell[42923, 1502, 98, 3, 22, "Input"],
Cell[43024, 1507, 173, 5, 26, "Text"],
Cell[43200, 1514, 579, 12, 79, "Input"],

Cell[CellGroupData[{
Cell[43804, 1530, 279, 6, 43, "Subsection",
  Evaluatable->False,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
Cell[44086, 1538, 301, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[44412, 1551, 212, 6, 22, "Input"],
Cell[44627, 1559, 1202, 33, 45, "Output"]
}, Open  ]],
Cell[45844, 1595, 563, 14, 54, "Text"],
Cell[46410, 1611, 177, 5, 22, "Input"],
Cell[46590, 1618, 420, 13, 40, "Text"],

Cell[CellGroupData[{
Cell[47035, 1635, 188, 5, 22, "Input"],
Cell[47226, 1642, 2350, 60, 72, "Output"]
}, Open  ]],
Cell[49591, 1705, 230, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[49846, 1717, 175, 5, 22, "Input"],
Cell[50024, 1724, 2337, 61, 72, "Output"]
}, Open  ]],
Cell[52376, 1788, 163, 5, 26, "Text"],
Cell[52542, 1795, 196, 5, 22, "Input"],
Cell[52741, 1802, 389, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[53155, 1822, 125, 3, 22, "Input"],
Cell[53283, 1827, 4756, 138, 46, "Output"]
}, Open  ]],
Cell[58054, 1968, 208, 8, 26, "Text"],
Cell[58265, 1978, 240, 8, 26, "Text"],
Cell[58508, 1988, 187, 5, 22, "Input"],
Cell[58698, 1995, 255, 10, 26, "Text"],
Cell[58956, 2007, 171, 4, 22, "Input"],
Cell[59130, 2013, 211, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[59366, 2025, 148, 4, 34, "Input"],
Cell[59517, 2031, 1054, 27, 23, "Output"]
}, Open  ]],
Cell[60586, 2061, 384, 14, 26, "Text"],

Cell[CellGroupData[{
Cell[60995, 2079, 136, 3, 22, "Input"],
Cell[61134, 2084, 316, 10, 22, "Output"]
}, Open  ]],
Cell[61465, 2097, 65, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[61555, 2103, 976, 24, 47, "Input"],
Cell[62534, 2129, 310, 10, 22, "Output"]
}, Open  ]],
Cell[62859, 2142, 509, 18, 40, "Text"],

Cell[CellGroupData[{
Cell[63393, 2164, 122, 3, 22, "Input"],
Cell[63518, 2169, 315, 10, 24, "Output"]
}, Open  ]],
Cell[63848, 2182, 393, 15, 26, "Text"],

Cell[CellGroupData[{
Cell[64266, 2201, 129, 3, 22, "Input"],
Cell[64398, 2206, 527, 13, 36, "Output"]
}, Open  ]],
Cell[64940, 2222, 285, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[65250, 2237, 223, 5, 34, "Input"],
Cell[65476, 2244, 317, 7, 28, "Output"]
}, Open  ]],
Cell[65808, 2254, 217, 7, 26, "Text"],

Cell[CellGroupData[{
Cell[66050, 2265, 174, 5, 22, "Input"],
Cell[66227, 2272, 348, 8, 43, "Output"]
}, Open  ]],
Cell[66590, 2283, 438, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[67053, 2303, 162, 4, 34, "Input"],
Cell[67218, 2309, 2661, 63, 63, "Output"]
}, Open  ]],
Cell[69894, 2375, 316, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[70235, 2390, 136, 3, 22, "Input"],
Cell[70374, 2395, 665, 18, 28, "Output"]
}, Open  ]],
Cell[71054, 2416, 405, 17, 26, "Text"],

Cell[CellGroupData[{
Cell[71484, 2437, 191, 5, 22, "Input"],
Cell[71678, 2444, 406, 11, 28, "Output"]
}, Open  ]],
Cell[72099, 2458, 162, 7, 26, "Text"],
Cell[72264, 2467, 212, 5, 32, "Input"],
Cell[72479, 2474, 173, 7, 26, "Text"],
Cell[72655, 2483, 123, 3, 22, "Input"],
Cell[72781, 2488, 176, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[72982, 2500, 160, 5, 22, "Input"],
Cell[73145, 2507, 494, 13, 28, "Output"]
}, Open  ]],
Cell[73654, 2523, 244, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[73923, 2535, 115, 3, 22, "Input"],
Cell[74041, 2540, 241, 8, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[74319, 2553, 117, 3, 22, "Input"],
Cell[74439, 2558, 389, 11, 28, "Output"]
}, Open  ]],
Cell[74843, 2572, 447, 18, 26, "Text"],

Cell[CellGroupData[{
Cell[75315, 2594, 212, 5, 22, "Input"],
Cell[75530, 2601, 218, 5, 28, "Output"]
}, Open  ]],
Cell[75763, 2609, 690, 27, 40, "Text"],

Cell[CellGroupData[{
Cell[76478, 2640, 506, 10, 86, "Input"],
Cell[76987, 2652, 585, 13, 49, "Output"]
}, Open  ]],
Cell[77587, 2668, 169, 5, 26, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[77793, 2678, 278, 6, 43, "Subsection",
  Evaluatable->False,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
Cell[78074, 2686, 372, 12, 40, "Text"],

Cell[CellGroupData[{
Cell[78471, 2702, 183, 4, 22, "Input"],
Cell[78657, 2708, 568, 17, 22, "Output"]
}, Open  ]],
Cell[79240, 2728, 286, 8, 40, "Text"],
Cell[79529, 2738, 709, 19, 34, "Input"],
Cell[80241, 2759, 512, 19, 40, "Text"],
Cell[80756, 2780, 237, 6, 22, "Input"],
Cell[80996, 2788, 533, 20, 40, "Text"],
Cell[81532, 2810, 202, 5, 22, "Input"],
Cell[81737, 2817, 226, 8, 26, "Text"],
Cell[81966, 2827, 215, 5, 22, "Input"],
Cell[82184, 2834, 163, 5, 26, "Text"],
Cell[82350, 2841, 308, 7, 48, "Input"],
Cell[82661, 2850, 389, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[83075, 2870, 218, 5, 47, "Input"],
Cell[83296, 2877, 1206, 38, 31, "Output"],
Cell[84505, 2917, 2243, 67, 32, "Output"],
Cell[86751, 2986, 3162, 91, 32, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[89950, 3082, 108, 2, 34, "Subsubsection"],
Cell[90061, 3086, 325, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[90411, 3099, 125, 3, 22, "Input"],
Cell[90539, 3104, 638, 18, 22, "Output"]
}, Open  ]],
Cell[91192, 3125, 437, 15, 26, "Text"],

Cell[CellGroupData[{
Cell[91654, 3144, 217, 5, 22, "Input"],
Cell[91874, 3151, 101, 3, 22, "Output"]
}, Open  ]],
Cell[91990, 3157, 254, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[92269, 3170, 126, 3, 22, "Input"],
Cell[92398, 3175, 102, 3, 22, "Output"]
}, Open  ]],
Cell[92515, 3181, 150, 3, 26, "Text"],
Cell[92668, 3186, 142, 4, 22, "Input"],
Cell[92813, 3192, 270, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[93108, 3205, 175, 5, 22, "Input"],
Cell[93286, 3212, 87, 3, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[93410, 3220, 193, 5, 22, "Input"],
Cell[93606, 3227, 87, 3, 22, "Output"]
}, Open  ]],
Cell[93708, 3233, 195, 5, 26, "Text"],

Cell[CellGroupData[{
Cell[93928, 3242, 130, 3, 22, "Input"],
Cell[94061, 3247, 149, 4, 22, "Output"]
}, Open  ]],
Cell[94225, 3254, 310, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[94560, 3269, 390, 9, 47, "Input"],
Cell[94953, 3280, 388, 10, 33, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[95390, 3296, 94, 2, 34, "Subsubsection"],
Cell[95487, 3300, 341, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[95853, 3316, 159, 4, 34, "Input"],
Cell[96015, 3322, 1985, 50, 38, "Output"]
}, Open  ]],
Cell[98015, 3375, 487, 17, 40, "Text"],

Cell[CellGroupData[{
Cell[98527, 3396, 155, 4, 22, "Input"],
Cell[98685, 3402, 546, 17, 22, "Output"]
}, Open  ]],
Cell[99246, 3422, 347, 9, 40, "Text"],

Cell[CellGroupData[{
Cell[99618, 3435, 442, 12, 22, "Input"],
Cell[100063, 3449, 310, 10, 22, "Output"]
}, Open  ]],
Cell[100388, 3462, 89, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[100502, 3468, 114, 3, 22, "Input"],
Cell[100619, 3473, 315, 10, 24, "Output"]
}, Open  ]],
Cell[100949, 3486, 208, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[101182, 3498, 159, 4, 34, "Input"],
Cell[101344, 3504, 790, 22, 24, "Output"]
}, Open  ]],
Cell[102149, 3529, 108, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[102282, 3535, 166, 4, 22, "Input"],
Cell[102451, 3541, 244, 5, 29, "Message"],
Cell[102698, 3548, 1197, 28, 53, "Output"]
}, Open  ]],
Cell[103910, 3579, 803, 26, 55, "Text"],

Cell[CellGroupData[{
Cell[104738, 3609, 320, 8, 60, "Input"],
Cell[105061, 3619, 482, 10, 48, "Output"]
}, Open  ]],
Cell[105558, 3632, 162, 5, 26, "Text"],

Cell[CellGroupData[{
Cell[105745, 3641, 282, 7, 34, "Input"],
Cell[106030, 3650, 566, 12, 64, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[106645, 3668, 93, 2, 34, "Subsubsection"],
Cell[106741, 3672, 262, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[107028, 3685, 176, 4, 34, "Input"],
Cell[107207, 3691, 4379, 98, 99, "Output"]
}, Open  ]],
Cell[111601, 3792, 368, 11, 40, "Text"],

Cell[CellGroupData[{
Cell[111994, 3807, 155, 4, 22, "Input"],
Cell[112152, 3813, 723, 20, 28, "Output"]
}, Open  ]],
Cell[112890, 3836, 112, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[113027, 3842, 135, 3, 22, "Input"],
Cell[113165, 3847, 503, 14, 28, "Output"]
}, Open  ]],
Cell[113683, 3864, 190, 5, 26, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[113922, 3875, 281, 6, 43, "Subsection",
  Evaluatable->False,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
Cell[114206, 3883, 277, 9, 26, "Text"],
Cell[114486, 3894, 206, 5, 22, "Input"],
Cell[114695, 3901, 88, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[114808, 3907, 183, 5, 22, "Input"],
Cell[114994, 3914, 824, 22, 28, "Output"]
}, Open  ]],
Cell[115833, 3939, 233, 8, 26, "Text"],
Cell[116069, 3949, 220, 5, 22, "Input"],
Cell[116292, 3956, 214, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[116531, 3968, 149, 4, 22, "Input"],
Cell[116683, 3974, 321, 9, 22, "Output"]
}, Open  ]],
Cell[117019, 3986, 524, 19, 40, "Text"],

Cell[CellGroupData[{
Cell[117568, 4009, 198, 5, 22, "Input"],
Cell[117769, 4016, 594, 13, 48, "Output"]
}, Open  ]],
Cell[118378, 4032, 286, 8, 40, "Text"],
Cell[118667, 4042, 534, 14, 22, "Input"],
Cell[119204, 4058, 182, 7, 26, "Text"],
Cell[119389, 4067, 294, 8, 23, "Input"],
Cell[119686, 4077, 537, 20, 40, "Text"],
Cell[120226, 4099, 202, 5, 22, "Input"],
Cell[120431, 4106, 131, 3, 26, "Text"],
Cell[120565, 4111, 186, 5, 22, "Input"],
Cell[120754, 4118, 163, 5, 26, "Text"],
Cell[120920, 4125, 208, 5, 22, "Input"],
Cell[121131, 4132, 389, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[121545, 4152, 125, 3, 22, "Input"],
Cell[121673, 4157, 6306, 175, 53, "Output"]
}, Open  ]],
Cell[127994, 4335, 264, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[128283, 4347, 171, 5, 22, "Input"],
Cell[128457, 4354, 132, 3, 22, "Output"]
}, Open  ]],
Cell[128604, 4360, 89, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[128718, 4366, 110, 3, 22, "Input"],
Cell[128831, 4371, 164, 4, 22, "Output"]
}, Open  ]],
Cell[129010, 4378, 248, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[129283, 4390, 155, 4, 34, "Input"],
Cell[129441, 4396, 752, 20, 28, "Output"]
}, Open  ]],
Cell[130208, 4419, 294, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[130527, 4434, 142, 4, 22, "Input"],
Cell[130672, 4440, 298, 10, 22, "Output"]
}, Open  ]],
Cell[130985, 4453, 65, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[131075, 4459, 417, 12, 22, "Input"],
Cell[131495, 4473, 310, 10, 22, "Output"]
}, Open  ]],
Cell[131820, 4486, 271, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[132116, 4498, 192, 5, 22, "Input"],
Cell[132311, 4505, 368, 8, 47, "Output"]
}, Open  ]],
Cell[132694, 4516, 89, 2, 26, "Text"],
Cell[132786, 4520, 116, 3, 22, "Input"],
Cell[132905, 4525, 262, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[133192, 4538, 205, 5, 34, "Input"],
Cell[133400, 4545, 2276, 52, 69, "Output"]
}, Open  ]],
Cell[135691, 4600, 320, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[136036, 4615, 170, 4, 34, "Input"],
Cell[136209, 4621, 433, 12, 28, "Output"]
}, Open  ]],
Cell[136657, 4636, 233, 8, 26, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[136939, 4650, 266, 6, 46, "Section",
  Evaluatable->False,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
Cell[137208, 4658, 405, 14, 26, "Text"],
Cell[137616, 4674, 133, 3, 22, "Input"],
Cell[137752, 4679, 146, 3, 26, "Text"],

Cell[CellGroupData[{
Cell[137923, 4686, 99, 3, 22, "Input"],
Cell[138025, 4691, 249, 7, 22, "Output"]
}, Open  ]],
Cell[138289, 4701, 211, 9, 26, "Text"],
Cell[138503, 4712, 137, 3, 22, "Input"],
Cell[138643, 4717, 361, 12, 40, "Text"],

Cell[CellGroupData[{
Cell[139029, 4733, 117, 3, 22, "Input"],
Cell[139149, 4738, 709, 19, 22, "Output"]
}, Open  ]],
Cell[139873, 4760, 282, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[140180, 4776, 174, 5, 22, "Input"],
Cell[140357, 4783, 529, 14, 22, "Output"]
}, Open  ]],
Cell[140901, 4800, 321, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[141247, 4816, 117, 3, 22, "Input"],
Cell[141367, 4821, 718, 19, 22, "Output"]
}, Open  ]],
Cell[142100, 4843, 367, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[142492, 4863, 183, 5, 22, "Input"],
Cell[142678, 4870, 694, 17, 46, "Output"]
}, Open  ]],
Cell[143387, 4890, 491, 19, 26, "Text"],

Cell[CellGroupData[{
Cell[143903, 4913, 244, 6, 22, "Input"],
Cell[144150, 4921, 773, 20, 34, "Output"]
}, Open  ]],
Cell[144938, 4944, 480, 15, 40, "Text"],

Cell[CellGroupData[{
Cell[145443, 4963, 127, 3, 22, "Input"],
Cell[145573, 4968, 555, 13, 41, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[146165, 4986, 129, 3, 22, "Input"],
Cell[146297, 4991, 609, 14, 41, "Output"]
}, Open  ]],
Cell[146921, 5008, 712, 25, 40, "Text"],

Cell[CellGroupData[{
Cell[147658, 5037, 121, 3, 22, "Input"],
Cell[147782, 5042, 205, 6, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[148024, 5053, 125, 3, 22, "Input"],
Cell[148152, 5058, 260, 7, 28, "Output"]
}, Open  ]],
Cell[148427, 5068, 1659, 50, 112, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[150123, 5123, 278, 6, 46, "Section",
  Evaluatable->False,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
Cell[150404, 5131, 453, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[150882, 5151, 195, 6, 22, "Input"],
Cell[151080, 5159, 503, 12, 41, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[151620, 5176, 239, 6, 22, "Input"],
Cell[151862, 5184, 568, 13, 41, "Output"]
}, Open  ]],
Cell[152445, 5200, 303, 12, 26, "Text"],

Cell[CellGroupData[{
Cell[152773, 5216, 180, 6, 22, "Input"],
Cell[152956, 5224, 199, 5, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[153192, 5234, 207, 7, 22, "Input"],
Cell[153402, 5243, 219, 5, 22, "Output"]
}, Open  ]],
Cell[153636, 5251, 261, 10, 26, "Text"],

Cell[CellGroupData[{
Cell[153922, 5265, 163, 5, 22, "Input"],
Cell[154088, 5272, 280, 8, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[154405, 5285, 194, 6, 22, "Input"],
Cell[154602, 5293, 319, 8, 22, "Output"]
}, Open  ]],
Cell[154936, 5304, 920, 36, 40, "Text"],
Cell[155859, 5342, 663, 25, 40, "Text"],

Cell[CellGroupData[{
Cell[156547, 5371, 211, 5, 22, "Input"],
Cell[156761, 5378, 1410, 37, 35, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[158208, 5420, 215, 5, 22, "Input"],
Cell[158426, 5427, 1469, 37, 35, "Output"]
}, Open  ]],
Cell[159910, 5467, 638, 23, 40, "Text"],

Cell[CellGroupData[{
Cell[160573, 5494, 205, 5, 22, "Input"],
Cell[160781, 5501, 842, 17, 74, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[161660, 5523, 209, 5, 22, "Input"],
Cell[161872, 5530, 838, 18, 73, "Output"]
}, Open  ]],
Cell[162725, 5551, 504, 20, 26, "Text"],

Cell[CellGroupData[{
Cell[163254, 5575, 205, 5, 22, "Input"],
Cell[163462, 5582, 2156, 52, 103, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[165655, 5639, 212, 6, 22, "Input"],
Cell[165870, 5647, 2151, 50, 103, "Output"]
}, Open  ]],
Cell[168036, 5700, 466, 15, 40, "Text"],
Cell[168505, 5717, 153, 4, 22, "Input"],
Cell[168661, 5723, 99, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[168785, 5729, 138, 3, 22, "Input"],
Cell[168926, 5734, 926, 26, 48, "Output"]
}, Open  ]],
Cell[169867, 5763, 485, 18, 40, "Text"],

Cell[CellGroupData[{
Cell[170377, 5785, 244, 6, 22, "Input"],
Cell[170624, 5793, 680, 18, 28, "Output"]
}, Open  ]],
Cell[171319, 5814, 406, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[171750, 5834, 274, 6, 47, "Input"],
Cell[172027, 5842, 267, 6, 28, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[172331, 5853, 289, 6, 47, "Input"],
Cell[172623, 5861, 261, 6, 28, "Output"]
}, Open  ]],
Cell[172899, 5870, 89, 2, 26, "Text"],
Cell[172991, 5874, 122, 3, 22, "Input"],
Cell[173116, 5879, 383, 13, 40, "Text"],

Cell[CellGroupData[{
Cell[173524, 5896, 163, 5, 22, "Input"],
Cell[173690, 5903, 3124, 61, 228, "Output"]
}, Open  ]],
Cell[176829, 5967, 200, 5, 26, "Text"],

Cell[CellGroupData[{
Cell[177054, 5976, 169, 5, 22, "Input"],
Cell[177226, 5983, 825, 14, 89, "Output"]
}, Open  ]],
Cell[178066, 6000, 605, 20, 40, "Text"],

Cell[CellGroupData[{
Cell[178696, 6024, 207, 5, 22, "Input"],
Cell[178906, 6031, 243, 6, 28, "Output"]
}, Open  ]],
Cell[179164, 6040, 428, 12, 40, "Text"],

Cell[CellGroupData[{
Cell[179617, 6056, 129, 3, 22, "Input"],
Cell[179749, 6061, 264, 7, 28, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[180050, 6073, 133, 3, 22, "Input"],
Cell[180186, 6078, 263, 7, 28, "Output"]
}, Open  ]],
Cell[180464, 6088, 532, 21, 40, "Text"],

Cell[CellGroupData[{
Cell[181021, 6113, 118, 3, 22, "Input"],
Cell[181142, 6118, 213, 6, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[181392, 6129, 122, 3, 22, "Input"],
Cell[181517, 6134, 351, 9, 28, "Output"]
}, Open  ]],
Cell[181883, 6146, 197, 7, 26, "Text"],

Cell[CellGroupData[{
Cell[182105, 6157, 193, 5, 22, "Input"],
Cell[182301, 6164, 252, 6, 28, "Output"]
}, Open  ]],
Cell[182568, 6173, 333, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[182926, 6188, 187, 5, 22, "Input"],
Cell[183116, 6195, 496, 13, 49, "Output"]
}, Open  ]],
Cell[183627, 6211, 442, 13, 40, "Text"],

Cell[CellGroupData[{
Cell[184094, 6228, 198, 5, 22, "Input"],
Cell[184295, 6235, 279, 7, 28, "Output"]
}, Open  ]],
Cell[184589, 6245, 270, 9, 26, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[184896, 6259, 286, 6, 46, "Section",
  Evaluatable->False,
  CellTags->{"ChapterTwo", "ChapterTwo.2", "ChapterTwo.2.1"}],
Cell[185185, 6267, 522, 14, 40, "Text"],
Cell[185710, 6283, 134, 3, 22, "Input"],
Cell[185847, 6288, 1050, 33, 69, "Text"],

Cell[CellGroupData[{
Cell[186922, 6325, 223, 6, 22, "Input"],
Cell[187148, 6333, 238, 6, 22, "Output"]
}, Open  ]],
Cell[187401, 6342, 529, 17, 40, "Text"],

Cell[CellGroupData[{
Cell[187955, 6363, 138, 3, 22, "Input"],
Cell[188096, 6368, 263, 7, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[188396, 6380, 177, 5, 22, "Input"],
Cell[188576, 6387, 305, 8, 22, "Output"]
}, Open  ]],
Cell[188896, 6398, 1006, 36, 40, "Text"],

Cell[CellGroupData[{
Cell[189927, 6438, 200, 5, 22, "Input"],
Cell[190130, 6445, 396, 10, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[190563, 6460, 204, 5, 22, "Input"],
Cell[190770, 6467, 413, 10, 22, "Output"]
}, Open  ]],
Cell[191198, 6480, 224, 7, 26, "Text"],
Cell[191425, 6489, 614, 21, 40, "Text"],

Cell[CellGroupData[{
Cell[192064, 6514, 291, 6, 47, "Input"],
Cell[192358, 6522, 2571, 59, 95, "Output"]
}, Open  ]],
Cell[194944, 6584, 225, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[195194, 6596, 183, 5, 22, "Input"],
Cell[195380, 6603, 2232, 51, 71, "Output"]
}, Open  ]],
Cell[197627, 6657, 278, 9, 26, "Text"],

Cell[CellGroupData[{
Cell[197930, 6670, 121, 3, 22, "Input"],
Cell[198054, 6675, 653, 16, 28, "Output"]
}, Open  ]],
Cell[198722, 6694, 295, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[199042, 6709, 240, 6, 34, "Input"],
Cell[199285, 6717, 131, 3, 22, "Output"]
}, Open  ]],
Cell[199431, 6723, 221, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[199677, 6735, 214, 5, 22, "Input"],
Cell[199894, 6742, 170, 4, 28, "Output"]
}, Open  ]],
Cell[200079, 6749, 105, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[200209, 6755, 166, 5, 22, "Input"],
Cell[200378, 6762, 196, 6, 28, "Output"]
}, Open  ]],
Cell[200589, 6771, 378, 16, 26, "Text"],

Cell[CellGroupData[{
Cell[200992, 6791, 128, 3, 22, "Input"],
Cell[201123, 6796, 570, 14, 53, "Output"]
}, Open  ]],
Cell[201708, 6813, 292, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[202025, 6828, 201, 6, 22, "Input"],
Cell[202229, 6836, 214, 4, 29, "Output"]
}, Open  ]],
Cell[202458, 6843, 66, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[202549, 6849, 150, 4, 22, "Input"],
Cell[202702, 6855, 212, 6, 28, "Output"]
}, Open  ]],
Cell[202929, 6864, 459, 20, 26, "Text"],

Cell[CellGroupData[{
Cell[203413, 6888, 127, 3, 22, "Input"],
Cell[203543, 6893, 205, 6, 22, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[203785, 6904, 145, 4, 22, "Input"],
Cell[203933, 6910, 320, 8, 28, "Output"]
}, Open  ]],
Cell[204268, 6921, 200, 5, 40, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[204505, 6931, 96, 2, 46, "Section"],
Cell[204604, 6935, 366, 10, 40, "Text"],
Cell[204973, 6947, 689, 15, 55, "Input"],
Cell[205665, 6964, 159, 5, 26, "Text"],
Cell[205827, 6971, 124, 3, 22, "Input"],

Cell[CellGroupData[{
Cell[205976, 6978, 460, 13, 48, "Input"],
Cell[206439, 6993, 484, 12, 33, "Output"]
}, Open  ]],
Cell[206938, 7008, 109, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[207072, 7014, 189, 5, 22, "Input"],
Cell[207264, 7021, 324, 9, 22, "Output"]
}, Open  ]],
Cell[207603, 7033, 110, 2, 26, "Text"],
Cell[207716, 7037, 213, 4, 22, "Input"],
Cell[207932, 7043, 621, 23, 40, "Text"],
Cell[208556, 7068, 251, 6, 34, "Input"],
Cell[208810, 7076, 225, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[209060, 7088, 174, 5, 22, "Input"],
Cell[209237, 7095, 2957, 65, 113, "Output"]
}, Open  ]],

Cell[CellGroupData[{
Cell[212231, 7165, 90, 2, 43, "Subsection"],
Cell[212324, 7169, 588, 17, 55, "Text"],

Cell[CellGroupData[{
Cell[212937, 7190, 166, 5, 22, "Input"],
Cell[213106, 7197, 137, 3, 22, "Output"]
}, Open  ]],
Cell[213258, 7203, 211, 8, 26, "Text"],
Cell[213472, 7213, 158, 4, 22, "Input"],
Cell[213633, 7219, 111, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[213769, 7225, 233, 6, 22, "Input"],
Cell[214005, 7233, 83, 3, 22, "Output"]
}, Open  ]],
Cell[214103, 7239, 188, 5, 26, "Text"],

Cell[CellGroupData[{
Cell[214316, 7248, 128, 3, 22, "Input"],
Cell[214447, 7253, 137, 3, 22, "Output"]
}, Open  ]],
Cell[214599, 7259, 120, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[214744, 7265, 134, 3, 22, "Input"],
Cell[214881, 7270, 144, 4, 22, "Output"]
}, Open  ]],
Cell[215040, 7277, 187, 7, 26, "Text"],

Cell[CellGroupData[{
Cell[215252, 7288, 345, 9, 35, "Input"],
Cell[215600, 7299, 351, 8, 22, "Output"]
}, Open  ]],
Cell[215966, 7310, 447, 15, 40, "Text"],

Cell[CellGroupData[{
Cell[216438, 7329, 202, 5, 34, "Input"],
Cell[216643, 7336, 235, 5, 26, "Output"]
}, Open  ]],
Cell[216893, 7344, 288, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[217206, 7359, 197, 6, 22, "Input"],
Cell[217406, 7367, 98, 3, 22, "Output"]
}, Open  ]],
Cell[217519, 7373, 189, 6, 26, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[217745, 7384, 90, 2, 43, "Subsection"],
Cell[217838, 7388, 145, 4, 26, "Text"],

Cell[CellGroupData[{
Cell[218008, 7396, 218, 5, 34, "Input"],
Cell[218229, 7403, 1162, 26, 62, "Output"]
}, Open  ]],
Cell[219406, 7432, 217, 8, 26, "Text"],

Cell[CellGroupData[{
Cell[219648, 7444, 171, 5, 22, "Input"],
Cell[219822, 7451, 175, 4, 22, "Output"]
}, Open  ]],
Cell[220012, 7458, 114, 2, 26, "Text"],

Cell[CellGroupData[{
Cell[220151, 7464, 233, 6, 22, "Input"],
Cell[220387, 7472, 109, 3, 22, "Output"]
}, Open  ]],
Cell[220511, 7478, 155, 5, 26, "Text"],

Cell[CellGroupData[{
Cell[220691, 7487, 128, 3, 22, "Input"],
Cell[220822, 7492, 142, 4, 22, "Output"]
}, Open  ]],
Cell[220979, 7499, 132, 4, 26, "Text"],

Cell[CellGroupData[{
Cell[221136, 7507, 135, 3, 22, "Input"],
Cell[221274, 7512, 147, 4, 22, "Output"]
}, Open  ]],
Cell[221436, 7519, 191, 7, 26, "Text"],

Cell[CellGroupData[{
Cell[221652, 7530, 345, 9, 35, "Input"],
Cell[222000, 7541, 366, 9, 22, "Output"]
}, Open  ]],
Cell[222381, 7553, 459, 15, 40, "Text"],

Cell[CellGroupData[{
Cell[222865, 7572, 195, 5, 22, "Input"],
Cell[223063, 7579, 359, 8, 28, "Output"]
}, Open  ]],
Cell[223437, 7590, 281, 11, 26, "Text"],

Cell[CellGroupData[{
Cell[223743, 7605, 180, 6, 22, "Input"],
Cell[223926, 7613, 206, 5, 29, "Output"]
}, Open  ]],
Cell[224147, 7621, 483, 19, 26, "Text"],

Cell[CellGroupData[{
Cell[224655, 7644, 414, 10, 47, "Input"],
Cell[225072, 7656, 424, 11, 28, "Output"]
}, Open  ]],
Cell[225511, 7670, 149, 5, 26, "Text"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

